<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\UserGameStats;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Carbon\Carbon;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;


class UserController extends Controller
{

    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'nullable|string',
            'fullname' => 'nullable|string|max:255', // Needed for new Google users
            'is_google_auth' => 'required|boolean',  // Differentiates login type
            'fcm_token' => 'nullable|string',  
        ]);

        $user = User::where('email', $request->email)->first();

        if ($user) {

            // 🔹 Check if user is active
            if ($user->status !== 'active') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Your account is not active.',
                ], 401);
            }

            // 🔹 If user exists
            if (!$request->is_google_auth) {
                // 🔹 Manual login → Validate password
                if (!Hash::check($request->password, $user->password)) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Invalid credentials.',
                    ], 401);
                }
            }

            // 🔹 Update FCM Token if sent
            if ($request->fcm_token) {
                $user->update([
                    'fcm_token' => $request->fcm_token,
                ]);
            }

            // 🔹 Retrieve Game Stats
            $gameStats = UserGameStats::where('user_id', $user->id)->get();

            // 🔹 Generate Token & Log in
            $token = $user->createToken('auth_token')->plainTextToken;

            return response()->json([
                'status' => 'success',
                'message' => 'Login successful',
                'token' => $token,
                'user' => $user,
                'game_stats' => $gameStats,
            ], 200);
        }

        // 🔹 If user doesn't exist & it's Google login → Auto-register
        if ($request->is_google_auth) {
            $user = User::create([
                'fullname' => $request->fullname ?? "Google User",
                'email' => $request->email,
                'password' => Hash::make(Str::random(20)), // Random password (not used)
                'overall_level' => 1,
                'current_points' => 0,
                'max_points' => 1000,
            ]);

            // 🔹 Initialize Game Stats
            $defaultGameStats = [
                [
                    'user_id' => $user->id,
                    'game_name' => 'numberlee_equation_solver_stats',
                    'games_played' => 0,
                    'games_won' => 0,
                    'best_try' => null,
                    'current_streak' => 0,
                    'max_streak' => 0,
                    'last_played_at' => null,
                    'level' => 1,
                ]
            ];

            foreach ($defaultGameStats as $game) {
                UserGameStats::create($game);
            }

            // 🔹 Generate Token & Log in
            $token = $user->createToken('auth_token')->plainTextToken;

            return response()->json([
                'status' => 'success',
                'message' => 'User registered and logged in successfully',
                'token' => $token,
                'user' => $user,
                'game_stats' => $defaultGameStats,
            ], 201);
        }

        // 🔹 If manual login & user doesn't exist → Show error
        return response()->json([
            'status' => 'error',
            'message' => 'User not found. Please sign up.',
        ], 404);
    }



    /**
     * Logout User (Invalidate Token)
     */
    public function logout(Request $request)
    {
        $user = $request->user();
        
        if ($user) {
            $user->tokens()->delete(); 
            $user->fcm_token = null;
            $user->save();     
        }

        return response()->json(['status' => 'success', 'message' => 'Logged out successfully']);
    }


    /**
     * Get all users.
     */
    public function getUsers()
    {
        $users = User::where('role', '!=', 'admin')->get();
        return response()->json($users);
    }

    /**
     * Get a single user by ID.
     */
    public function getUserById($id)
    {
        $user = User::find($id);
        if (!$user) {
            return response()->json(['message' => 'User not found'], 404);
        }

        return response()->json($user);
    }

    /**
     * Create a new user.
     */
    
     public function register(Request $request)
     {
         // ✅ Validate request
         $request->validate([
             'fullname' => 'required|string|max:255',
             'email' => 'required|email|unique:users,email',
             'password' => 'required|string|min:6|confirmed',
             'game_stats' => 'nullable|array',
         ]);
     
         $isGoogleSignup = $request->password === 'google_auth_Hv5BWNt';
     
         // ✅ Check if user already exists
         $user = User::where('email', $request->email)->first();
         if ($user) {
             if ($isGoogleSignup) {
                 // ✅ Google Signup → Auto Login
                 $token = $user->createToken('auth_token')->plainTextToken;
                
                 
                 return response()->json([
                     'status' => 'success',
                     'message' => 'User logged in successfully',
                     'token' => $token,
                     'user' => $user,
                     'game_stats' => $user->gameStats,
                 ], 200);
             } else {
                 return response()->json([
                     'status' => 'error',
                     'message' => 'Email already exists. Please log in instead.',
                 ], 422);
             }
         }
     
         try {
             // ✅ Start Transaction
             DB::beginTransaction();
     
             // ✅ Create User
             $user = User::create([
                 'fullname' => $request->fullname,
                 'email' => $request->email,
                 'password' => $isGoogleSignup ? Hash::make(Str::random(20)) : Hash::make($request->password),
                 'overall_level' => 1,
                 'current_points' => 0,
                 'max_points' => 1000,
             ]);
     
             // ✅ Process Game Stats (if provided)
             if ($request->filled('game_stats')) {
                 foreach ($request->game_stats as $game) {
                    $achievements = [];
                    if ($game['level'] >= 100) {
                        $achievements[] = 'gold';
                    } elseif ($game['level'] >= 50) {
                        $achievements[] = 'silver';
                    } elseif ($game['level'] >= 30) {
                        $achievements[] = 'bronze';
                    }

                     UserGameStats::create([
                         'user_id' => $user->id,
                         'game_name' => $game['game_name'],
                         'games_played' => $game['games_played'],
                         'games_won_easy' => $game['games_won_easy'],
                         'games_won_medium' => $game['games_won_medium'],
                         'games_won_hard' => $game['games_won_hard'],
                         'best_try' => $game['best_try'],
                         'current_streak' => $game['current_streak'],
                         'max_streak' => $game['max_streak'],
                         'last_played_at' => $game['last_played_at'] ? Carbon::parse($game['last_played_at'])->format('Y-m-d H:i:s') : null,
                         'level' => $game['level'],
                         'achivements' => $achievements,
                     ]);
     
                     // ✅ Calculate User's Points & Coins based on provided stats
                     $difficultyPoints = ['easy' => 50, 'medium' => 100, 'hard' => 150];
     
                     foreach (['easy', 'medium', 'hard'] as $difficulty) {
                         $gamesWonField = "games_won_{$difficulty}";
                         $user->current_points += $game[$gamesWonField] * $difficultyPoints[$difficulty];
                     }
                 }
             }
     
             // ✅ Check for Overall User Level-Up
             while ($user->current_points >= $user->max_points) {
                 $user->current_points -= $user->max_points; // Carry over extra points
                 $user->overall_level += 1;
                 $user->max_points += 500; // Increase threshold for next level
             }
     
             $user->save();
     
             // ✅ Commit Transaction (Everything is good)
             DB::commit();
     
             // ✅ Generate Token
             $token = $user->createToken('auth_token')->plainTextToken;
     
             return response()->json([
                 'status' => 'success',
                 'message' => 'User registered successfully',
                 'token' => $token,
                 'user' => $user,
                 'game_stats' => $user->gameStats,
             ], 201);
         } catch (\Exception $e) {
             // ❌ Rollback Transaction if Something Fails
             DB::rollBack();
             
             // Log error for debugging
             //Log::error('User registration failed: ' . $e->getMessage());
     
             return response()->json([
                 'status' => 'error',
                 'message' => 'Registration failed. Please try again.',
             ], 500);
         }
     }
     
     
    


    /**
     * Update an existing user.
     */
  

   public function updateUser(Request $request)
    {
        $user = Auth::user(); // Current authenticated user

        $request->validate([
            'fullname' => 'string|max:255',
            'oldPassword' => 'nullable|string',
            'newPassword' => 'nullable|string|min:6',
            'profile_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5120',
        ]);

        // Update full name
        if ($request->filled('fullname')) {
            $user->fullname = $request->fullname;
        }

        // Handle password change
        if ($request->filled('oldPassword') && $request->filled('newPassword')) {
            if (!Hash::check($request->oldPassword, $user->password)) {
                return response()->json([
                    'message' => 'Old password does not match our records.'
                ], 422);
            }

            $user->password = Hash::make($request->newPassword);
        }

       // Handle profile image upload
        if ($request->hasFile('profile_image')) {
            $oldImagePath = $user->getRawOriginal('profile_image');

            if ($oldImagePath) {
                $fullPath = public_path($oldImagePath);
                if (file_exists($fullPath)) {
                    unlink($fullPath);
                }
            }

            $image = $request->file('profile_image');
            $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('uploads/images/profile'), $imageName);
            $user->profile_image = 'uploads/images/profile/' . $imageName;
        }



        $user->save();

        return response()->json([
            'message' => 'User updated successfully',
            'user' => $user
        ]);
    }


    /**
     * Delete a user.
     */
    public function deleteUser($id)
    {
        $user = User::find($id);
        if (!$user) {
            return response()->json(['message' => 'User not found'], 404);
        }

        $user->delete();

        return response()->json(['message' => 'User deleted successfully']);
    }

    public function changeStatusUser($id)
    {
        $user = User::find($id);
        if (!$user) {
            return response()->json(['message' => 'User not found'], 404);
        }

        // Toggle status
        $user->status = $user->status === 'active' ? 'in-active' : 'active';
        $user->save();

        return response()->json(['message' => 'User status updated successfully']);
    }
}
