<?php

namespace App\Http\Controllers\AdminController;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Banner;
use App\Models\Notification;
use App\Models\Popup;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

class AuthController extends Controller
{
    public function showLogin()
    {
        return view('admin.login');
    }


    public function loginWeb(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required|string',
        ]);
    
        if (Auth::attempt($request->only('email', 'password'))) {
            $request->session()->regenerate();
    
            $user = Auth::user();
    
            // 🔒 Ensure user is admin
            if ($user->role !== 'admin') {
                Auth::logout();
                return back()->withErrors([
                    'email' => 'Access denied. Admins only.',
                ]);
            }
    
            // ✅ Generate token for API use
            $token = $user->createToken('admin_token')->plainTextToken;
            session(['admin_api_token' => $token]);
    
            return redirect()->route('admin.dashboard');
        }
    
        return back()->withErrors([
            'email' => 'Invalid credentials.',
        ]);
    }
    

    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect()->route('admin.login');
    }



    public function showProfile()
    {
        return view('admin.profileSetting');
    }
    
    public function updateProfile(Request $request)
    {
        $request->validate([
            'email' => 'required|email|unique:users,email,' . auth()->id(),
            'password' => 'nullable|min:6|confirmed'
        ]);
    
        $user = auth()->user();
        $user->email = $request->email;
        if ($request->filled('password')) {
            $user->password = bcrypt($request->password);
        }
        $user->save();
        
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return response()->json(['message' => 'Profile updated successfully']);
    }
    



    public function dashboard()
    {
        $now = Carbon::now();
        $startOfThisWeek = $now->startOfWeek();
        $startOfLastWeek = (clone $startOfThisWeek)->subWeek();
        $endOfLastWeek = (clone $startOfLastWeek)->endOfWeek();

        // Users
        $userCount = \App\Models\User::count();
        $thisWeekUsers = \App\Models\User::whereBetween('created_at', [$startOfThisWeek, $now])->count();
        $lastWeekUsers = \App\Models\User::whereBetween('created_at', [$startOfLastWeek, $endOfLastWeek])->count();
        $userGrowth = $this->calculateGrowth($thisWeekUsers, $lastWeekUsers);

        // Banners
        $bannerCount = \App\Models\Banner::count();
        $thisWeekBanners = \App\Models\Banner::whereBetween('created_at', [$startOfThisWeek, $now])->count();
        $lastWeekBanners = \App\Models\Banner::whereBetween('created_at', [$startOfLastWeek, $endOfLastWeek])->count();
        $bannerGrowth = $this->calculateGrowth($thisWeekBanners, $lastWeekBanners);

        // Notifications
        $notificationCount = \App\Models\Notification::count();
        $thisWeekNotifications = \App\Models\Notification::whereBetween('created_at', [$startOfThisWeek, $now])->count();
        $lastWeekNotifications = \App\Models\Notification::whereBetween('created_at', [$startOfLastWeek, $endOfLastWeek])->count();
        $notificationGrowth = $this->calculateGrowth($thisWeekNotifications, $lastWeekNotifications);

        // Popups
        $popupCount = \App\Models\Popup::count();
        $thisWeekPopups = \App\Models\Popup::whereBetween('created_at', [$startOfThisWeek, $now])->count();
        $lastWeekPopups = \App\Models\Popup::whereBetween('created_at', [$startOfLastWeek, $endOfLastWeek])->count();
        $popupGrowth = $this->calculateGrowth($thisWeekPopups, $lastWeekPopups);

        return view('admin.dashboard', compact(
            'userCount', 'bannerCount', 'notificationCount', 'popupCount',
            'userGrowth', 'bannerGrowth', 'notificationGrowth', 'popupGrowth'
        ));
    }

    private function calculateGrowth($current, $previous)
    {
        if ($previous == 0) {
            return $current > 0 ? ['percent' => 100, 'direction' => 'up'] : ['percent' => 0, 'direction' => 'none'];
        }

        $change = (($current - $previous) / $previous) * 100;

        return [
            'percent' => abs(round($change)),
            'direction' => $change >= 0 ? 'up' : 'down'
        ];
    }

}
