<?php

namespace App\Http\Controllers\Api;

use Exception;
use App\Models\Settings\Bonus;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Auth;
use DataTables;
class BonusController extends Controller
{
    public function get_bonuses_index()
{
    try {
        $org_id = Auth::user()->org_id;
        $bonuses = Bonus::where('org_id', $org_id)->with(['employee.userProfile.user'])->get();

        return DataTables::of($bonuses)
            ->addColumn('count', function($row) {
                static $count = 0;
                return ++$count;
            })
            ->addColumn('employee_name', function ($bonus) {
                return $bonus->employee->userProfile->user->name;
            })
            ->editColumn('date', function ($bonus) {
                return \Carbon\Carbon::parse($bonus->date)->format('d-M-Y');
            })
            ->make(true);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Something went wrong & ' . $e->getMessage(),
        ]);
    }
}

    
    public function get_bonuses()
    {
        try {
            $org_id = Auth::user()->org_id;
            $bonuses = Bonus::where('org_id',$org_id)->with(['employee.userProfile.user'])
                ->get()
                ->map(function ($bonus) {
                    return [
                        'id' => $bonus->id,
                        'org_id' => $bonus->org_id,
                        'employee_id' => $bonus->employee_id,
                        'amount' => $bonus->amount,
                        'date' => $bonus->date,
                        'reason' => $bonus->reason,
                        'status' => $bonus->status,
                        'employee_name' => $bonus->employee->userProfile->user->name ?? 'N/A',
                    ];
                });

            return response()->json([
                "message" => "Bonuses fetched successfully",
                "bonuses" => $bonuses
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch bonuses.",
                "details" => $e->getMessage()
            ], 500);
        }
    }

    // Add a new bonus
    public function add_bonus(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'org_id' => 'required|exists:organizations,id',
                'employee_id' => 'required|exists:employee,id',
                'amount' => 'required|numeric',
                'date' => 'required',
                'reason' => 'required|string',
                'status' => 'required|string|in:active,inactive',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $bonus = new Bonus();
            $bonus->org_id = $request->org_id;
            $bonus->employee_id = $request->employee_id;
            $bonus->amount = $request->amount;
            $bonus->date = $request->date;
            $bonus->reason = $request->reason;
            $bonus->status = $request->status;
            $bonus->save();

            return response()->json([
                'success' => true,
                'message' => 'Bonus created successfully',
                'bonus' => $bonus
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create bonus.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    // Fetch a single bonus by ID
    public function get_bonus($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $bonus = Bonus::where('org_id',$org_id)->with(['employee.userProfile.user', 'organization'])->findOrFail($id);

            return response()->json([
                'message' => 'Bonus retrieved successfully',
                'bonus' => $bonus
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Bonus not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve bonus.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    // Update an existing bonus
    public function update_bonus(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'employee_id' => 'required|exists:employee,id',
                'amount' => 'required|numeric',
                'date' => 'required',
                'reason' => 'required|string',
                'status' => 'required|string|in:active,inactive',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $bonus = Bonus::findOrFail($id);
            $bonus->employee_id = $request->employee_id;
            $bonus->amount = $request->amount;
            $bonus->date = $request->date;
            $bonus->reason = $request->reason;
            $bonus->status = $request->status;
            $bonus->save();

            return response()->json([
                'success' => true,
                'message' => 'Bonus updated successfully',
                'bonus' => $bonus
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Bonus not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update bonus',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    // Delete a bonus
    public function delete_bonus($id)
    {
        try {
            $bonus = Bonus::findOrFail($id);
            $bonus->delete();

            return response()->json(['message' => 'Bonus deleted successfully']);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => 'Bonus not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'Failed to delete bonus',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
