<?php

namespace App\Http\Controllers\Api;

use App\Models\Settings\LeaveType;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Exception;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use DataTables;
class LeaveTypeController extends Controller
{
    public function get_leave_types_index()
    {
        try {
            $org_id = Auth::user()->org_id;
            $leaveTypes = LeaveType::where('org_id', $org_id)->get();
    
            return DataTables::of($leaveTypes)
                ->addColumn('count', function ($row) {
                    static $count = 0;
                    return ++$count;
                })
                ->make(true);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong & ' . $e->getMessage(),
            ]);
        }
    }
    
    public function get_leave_types()
    {
        try {
            $org_id = Auth::user()->org_id;
            $leaveTypes = LeaveType::where('org_id', $org_id)->get();

            return response()->json([
                'message' => 'Leave types fetched successfully',
                'leave_types' => $leaveTypes
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to fetch leave types.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    // Add a new leave type
    public function add_leave_type(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'org_id' => 'required|exists:organizations,id',
                'name' => [
                    'required',
                    'string',
                    function ($attribute, $value, $fail) use ($request) {
                        if ($request->filled('org_id')) {
                            $exists = LeaveType::where('name', $value)
                                ->where('org_id', $request->org_id)
                                ->exists();
                            if ($exists) {
                                $fail('The name has already been taken for the specified organization.');
                            }
                        }
                    },
                ],
                'status' => 'required|string|in:active,inactive',
            ]);
            

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $leaveType = new LeaveType();
            $leaveType->name = $request->name;
            $leaveType->status = $request->status;
            $leaveType->org_id = $request->org_id;
            $leaveType->save();

            return response()->json([
                'success' => true,
                'message' => 'Leave type created successfully',
                'leave_type' => $leaveType
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create leave type.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    // Get a specific leave type by ID
    public function get_leave_type($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $leaveType = LeaveType::where('org_id', $org_id)->findOrFail($id);

            return response()->json([
                'message' => 'Leave type retrieved successfully',
                'leave_type' => $leaveType
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Leave type not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve leave type.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    // Update a specific leave type
    public function update_leave_type(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => [
                    'required',
                    'string',
                    function ($attribute, $value, $fail) use ($request, $id) {
                        $exists = LeaveType::where('name', $value)
                            ->where('org_id', Auth::user()->org_id)
                            ->where('id', '!=', $id)
                            ->exists();
                        if ($exists) {
                            $fail('The name has already been taken for the specified organization.');
                        }
                    },
                ],
                'status' => 'required|string|in:active,inactive',
            ]);
            

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $leaveType = LeaveType::findOrFail($id);
            $leaveType->name = $request->name;
            $leaveType->status = $request->status;
            $leaveType->save();

            return response()->json([
                'success' => true,
                'message' => 'Leave type updated successfully',
                'leave_type' => $leaveType
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Leave type not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update leave type',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    // Delete a specific leave type
    public function delete_leave_type($id)
    {
        try {
            $leaveType = LeaveType::findOrFail($id);
            $leaveType->delete();

            return response()->json(['message' => 'Leave type deleted successfully']);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => 'Leave type not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'Failed to delete leave type',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
