<?php

namespace App\Http\Controllers\Api;

use Exception;
use App\Models\Settings\TaxSlab;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Auth;
use DataTables;
class TaxSlabController extends Controller
{
    public function get_tax_slabs_index()
    {
        try {
            $org_id = Auth::user()->org_id;
            $taxSlabs = TaxSlab::where('org_id', $org_id)->get();
    
            return DataTables::of($taxSlabs)
                ->addColumn('count', function ($row) {
                    static $count = 0;
                    return ++$count;
                })
                ->make(true);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong & ' . $e->getMessage(),
            ]);
        }
    }
    
    public function get_tax_slabs()
    {
        try {
            $org_id = Auth::user()->org_id;
            $taxSlabs = TaxSlab::where('org_id',$org_id)->get();

            return response()->json([
                "message" => "Tax slabs fetched successfully",
                "tax_slabs" => $taxSlabs
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch tax slabs.",
                "details" => $e->getMessage()
            ], 500);
        }
    }

    public function add_tax_slab(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'tax_slab' => 'required|string',
                'tax' => 'required|string',
                'flat_tax' => 'required|string',
                'org_id' => 'required|exists:organizations,id',
                'status' => 'required|string|in:active,inactive',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $taxSlab = new TaxSlab();
            $taxSlab->tax_slab = $request->tax_slab;
            $taxSlab->tax = $request->tax;
            $taxSlab->flat_tax = $request->flat_tax;
            $taxSlab->org_id = $request->org_id;
            $taxSlab->status = $request->status;
            $taxSlab->save();

            return response()->json([
                'success' => true,
                'message' => 'Tax slab created successfully',
                'tax_slab' => $taxSlab
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create tax slab.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function get_tax_slab($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $taxSlab = TaxSlab::where('org_id',$org_id)->findOrFail($id);
            return response()->json([
                'message' => 'Tax slab retrieved successfully',
                'tax_slab' => $taxSlab
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Tax slab not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve tax slab.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function update_tax_slab(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'tax_slab' => 'required|string',
                'tax' => 'required|string',
                'flat_tax' => 'required|string',
                'status' => 'required|string|in:active,inactive',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $taxSlab = TaxSlab::findOrFail($id);

            $taxSlab->tax_slab = $request->tax_slab;
            $taxSlab->tax = $request->tax;
            $taxSlab->flat_tax = $request->flat_tax;
            $taxSlab->status = $request->status;
            $taxSlab->save();

            return response()->json([
                'success' => true,
                'message' => 'Tax slab updated successfully',
                'tax_slab' => $taxSlab
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Tax slab not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update tax slab',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function delete_tax_slab($id)
    {
        try {
            $taxSlab = TaxSlab::findOrFail($id);

            $taxSlab->delete();

            return response()->json(['message' => 'Tax slab deleted successfully']);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => 'Tax slab not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'Failed to delete tax slab',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
