<?php

namespace App\Http\Controllers\Api;

use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Mail\ResetPasswordMail;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Password;
use DataTables;
class UserController extends Controller
{
    public function user_list_index(Request $request)
{
    try {
        $query = User::query();

        // Search filters
        if ($request->has('search')) {
            $search = $request->search;
            $not = $request->has('not') ? $request->not : false;

            if ($not) {
                $query->where('id', '!=', Auth::user()->id)
                    ->where(function ($q) use ($search) {
                        $q->where('role', 'not like', '%' . $search . '%')
                            ->where('name', 'not like', '%' . $search . '%')
                            ->where('email', 'not like', '%' . $search . '%');
                    });
            } else {
                $query->where(function ($q) use ($search) {
                    $q->where('role', 'like', '%' . $search . '%')
                        ->orWhere('name', 'like', '%' . $search . '%')
                        ->orWhere('email', 'like', '%' . $search . '%');
                });
            }
        } else {
            $query->where('id', '!=', Auth::user()->id);
        }

        // Filter by organization ID
        if ($request->has('org_id')) {
            $query->where('org_id', $request->org_id);
        }

        // Filter by role for admin users
        if (Auth::user()->role == 'admin' && !$request->has('search')) {
            $query->where(function ($q) {
                $q->where('role', '=', 'admin_normal')
                  ->orWhere('role', '=', 'organization');
            });
        }

        // Fetch users with DataTables
        $users = $query->get();
        return DataTables::of($users)
            ->editColumn('role', function ($user) {
                return $user->role;
            })
            ->make(true);
    } catch (Exception $e) {
        return response()->json([
            "error" => "Failed to fetch Users.",
            "details" => $e->getMessage()
        ], 500);
    }
}

    public function user_list(Request $request)
    {
        try {
            $query = User::query();

            // Check if the search parameter is present
            if ($request->has('search')) {
                $search = $request->search;

                $not = false;
                if($request->has('not')){
                    $not = $request->not;
                }

                if ($not) {
                    // Exclude the authenticated user and filter out matching records
                    $query->where('id', '!=', Auth::user()->id)
                        ->where(function ($q) use ($search) {
                            $q->where('role', 'not like', '%' . $search . '%')
                                ->where('name', 'not like', '%' . $search . '%')
                                ->where('email', 'not like', '%' . $search . '%');
                        });
                } else {
                    // Otherwise, filter based on the search term
                    $query->where(function ($q) use ($search) {
                        $q->where('role', 'like', '%' . $search . '%')
                        ->orWhere('name', 'like', '%' . $search . '%')
                        ->orWhere('email', 'like', '%' . $search . '%');
                    });
                }
            } else {
                $query->where('id', '!=', Auth::user()->id);
            }

            // Filter by organization ID if provided
            if ($request->has('org_id')) {
                $org_id = $request->org_id;
                $query->where('org_id', $org_id);
            }

            if(Auth::user()->role == 'admin' && !$request->has('search')){
                $query->where('role', '=', 'admin_normal')->orWhere('role', '=', 'organization');
            }

            // Get the users
            $users = $query->get();

            return response()->json([
                "message" => "Users fetched successfully",
                "users" => $users
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch Users.",
                "details" => $e->getMessage()
            ], 500);
        }
    }

    public function get_assigned_permission_user() {
        
        $users = User::where('org_id',Auth::user()->org_id)->where('role', 'employee')->whereNotNull('permissions')->get();
        
        return response()->json([
            'message' => 'Users fetched successfully',
            'users' => $users
        ]);
    }
    public function get_assigned_permission_user_index() {
        try {
            $org_id = Auth::user()->org_id;
    
            // Fetch users with permissions
            $users = User::where('org_id', $org_id)
                ->where('role', 'employee')
                ->whereNotNull('permissions')
                ->get();
    
            return DataTables::of($users)
                ->addColumn('count', function ($row) {
                    static $count = 0;
                    return ++$count;
                })
                ->make(true);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong & ' . $e->getMessage(),
            ]);
        }
    }
    
    
    


    
    public function add_user_form(Request $request)
    {
        // Validate request data
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'permissions' => 'nullable|max:255',
            'email' => [
                'required',
                'email',
                'max:255',
                function ($attribute, $value, $fail) use ($request) {
                    if ($request->filled('org_id')) {
                        $exists = \App\Models\User::where('email', $value)
                            ->where('org_id', $request->org_id)
                            ->exists();
                        if ($exists) {
                            $fail('The email has already been taken for the specified organization.');
                        }
                    } else {
                        if (\App\Models\User::where('email', $value)->exists()) {
                            $fail('The email has already been taken.');
                        }
                    }
                },
            ],
            'password' => 'required|string|min:8|confirmed',
            'org_id' => 'nullable|exists:organizations,id',
            'role' => 'nullable|string|in:admin,normal,organization,employee,admin_normal',
        ]);
        
    
        try {
            // Create and save the new user
            $user = User::create([
                'name' => $validatedData['name'],
                'email' => $validatedData['email'],
                'org_id' => $validatedData['org_id'] ?? null,
                'permissions' => $request->permissions ?? null,
                'role' => $validatedData['role'] ?? 'normal',
                'password' => bcrypt($validatedData['password']),
            ]);
    
            // Return success response
            return response()->json([
                'success' => true,
                'message' => 'User created successfully',
                'user' => $user
            ], 201);
    
        } catch (\Exception $e) {
            // Handle exceptions
            return response()->json([
                'success' => false,
                'message' => 'Failed to create user.',
                'details' => $e->getMessage()
            ], 500);
        }
    }
    
    public function get_user($id)
    {
        try {
            
            $user = User::findOrFail($id);

            return response()->json([
                'message' => 'User retrieved successfully',
                'user' => $user
            ], 200);
        } catch (ModelNotFoundException $e) {
            // Return error response for not found
            return response()->json([
                'error' => 'User not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            // Return error response for other failures
            return response()->json([
                'error' => 'Failed to retrieve User.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function update_user_form(Request $request, $id)
    {
        try {
            $request->validate([
                'name' => 'nullable|string|max:255',
                'email' => 'nullable|string|email|max:255|unique:users,email,' . $id,
                'password' => 'nullable|string|min:8|confirmed',
                'role' => 'nullable|string',
                'permissions' => 'nullable',
            ]);

            $user = User::findOrFail($id);

            if ($request->filled('name')) {
                $user->name = $request->name;
            }
            if ($request->filled('email')) {
                $user->email = $request->email;
            }
            if ($request->filled('role')) {
                $user->role = $request->role;
            }
            if ($request->filled('permissions')) {
                $user->permissions = $request->permissions;
            }
            if ($request->filled('password')) {
                $user->password = bcrypt($request->password);
            }

            $user->save();

            return response()->json([
                'success' => true,
                'message' => 'User updated successfully',
                'user' => $user
            ], 200);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->validator->getMessageBag()
            ], 422);

        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'User not found',
            ], 404);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update User',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    

    public function delete_user($id)
    {
        $user = User::find($id);

        if (!$user) {
            return response()->json(['message' => 'User not found'], 404);
        }

        $user->delete();

        return response()->json(['message' => 'User deleted successfully']);
    }

    public function sendResetLinkEmail(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|exists:users,email',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::where('email', $request->email)->first();

        // Generate reset token
        $token = Password::createToken($user);

        // Send custom reset email
        Mail::to($request->email)->send(new ResetPasswordMail($token, $request->email));

        return response()->json([
            'success' => true,
            'message' => 'Reset password link sent to your email.',
        ], 200);
    }

    public function reset(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'token' => 'required',
            'email' => 'required|email|exists:users,email',
            'password' => 'required|string|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $status = Password::reset(
            $request->only('email', 'password', 'password_confirmation', 'token'),
            function ($user, $password) {
                $user->password = Hash::make($password);
                $user->save();
            }
        );

        if ($status === Password::PASSWORD_RESET) {
            return response()->json([
                'success' => true,
                'message' => __($status)
            ], 200);
        }

        return response()->json([
            'success' => false,
            'message' => __($status)
        ], 500);
    }    


    


}
