<?php

namespace App\Http\Controllers\Settings;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;
use App\DataTables\Settings\DocumentDataTable;

class DocumentController extends Controller
{
    public function document_list(DocumentDataTable $dataTable){
        return $dataTable->render('admin.settings.document');
    }
    public function add_document_form(Request $request)
    {
        // Validate the input data, including the document file
        $request->validate([
            'name' => 'required|string|max:255',
            'document' => 'required|file',
            'remind_date' => 'nullable',
        ]);
    
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $org_id = session('org_id');
            // Prepare the request by attaching the file and sending other data as form fields
            $response = Http::withToken($token)
            ->attach('document', file_get_contents($request->file('document')->getRealPath()), $request->file('document')->getClientOriginalName())
            ->post("{$apiBaseUrl}/documents/store", [
                'name' => $request->name,
                'remind_date' => $request->remind_date,
                'org_id' => $org_id
                ]);
    
            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Document added successfully']);
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to add Document.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }
    public function update_document_form(Request $request, $id)
    {
        // Validate the input data, including the document file
        $request->validate([
            'name' => 'required|string|max:255',
            'document' => 'required|file',
            'remind_date' => 'nullable',
        ]);
    
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            
            // Prepare the request by attaching the file and sending other data as form fields
            $response = Http::withToken($token)
                ->timeout(60) // Increase the timeout to 60 seconds
                ->attach('document', file_get_contents($request->file('document')->getRealPath()), $request->file('document')->getClientOriginalName())
                ->post("{$apiBaseUrl}/documents/update/{$id}", [
                    'name' => $request->name,
                    'remind_date' => $request->remind_date,
                    'status' => $request->status,
                ]);
    
            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Document Updated successfully']);
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update Document.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }
    

    public function delete_document($id)
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->delete("{$apiBaseUrl}/documents/{$id}");
    
        if ($response->successful()) {
            return response()->json(['success' => true, 'message' => 'Document deleted successfully']);
        } else {
            // Extract error message from the API response, defaulting to a generic message
            $error = $response->json();
            $errorMessage = $error['message'] ?? 'Unable to delete Document';
            
            return response()->json([
                'success' => false,
                'message' => $errorMessage
            ], $response->status());
        }
    }
    public function get_document($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $response = Http::withToken($token)->get("{$apiBaseUrl}/documents/get/{$id}");
    
          
    
            if ($response->successful()) {
                // Return the department data from the API response
                return response()->json($response->json()['document']);
            } else {
                return response()->json(['status' => 404, 'message' => 'Document not found.'], 404);
            }
        } catch (\Exception $e) {
            // Dump the exception message and die
            dd('An error occurred: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
    
}
