<?php

namespace App\Http\Controllers\SuperAdmin;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;
use App\DataTables\SuperAdmin\OrganizationDataTable;

class OrganizationController extends Controller
{
    private function fetch_users(): array
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->get("{$apiBaseUrl}/users?search=organization");

        if ($response->successful()) {
            $users = $response->json()['users'] ?? [];
            // Extract only the necessary columns
            $users = array_map(function ($users) {
                return [
                    'id' => $users['id'],
                    'name' => $users['name'],
                ];
            }, $users);

            return $users;
        }

        return [];
    }
    public function organization_list(OrganizationDataTable $dataTable)
    {
        $users = $this->fetch_users();
        return $dataTable->render('super_admin.organization.organization',compact('users'));
    }
    public function add_organization_form(Request $request)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
    
            // Prepare data to send, including the permissions array
            $response = Http::withToken($token)->post("{$apiBaseUrl}/organizations/store", [
                'user_id' => $request->user_id,
                'name' => $request->name,
                'permissions' => $request->permissions, // Ensure this is an array
                'status' => $request->status,  // Use status instead of $request->name
            ]);
    
            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Organization added successfully']);
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to add Organization.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }
    
    public function update_organization_form(Request $request, $id)
    {
        try {
            // Get the token from session and set the base API URL
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
    
            // Send a PUT request to update the department
            $response = Http::withToken($token)->put("{$apiBaseUrl}/organizations/update/{$id}", [
                'user_id' => $request->user_id,
                'name' => $request->name,
                'permissions' => $request->permissions, // Ensure this is an array
                'status' => $request->status,  // Use status instead of $request->name
            ]);
    
            // Check if the response is successful
            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Organization updated successfully']);
            } else {
                // If the response contains an error, extract the error message
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update Organization.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Handle general exceptions
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }
    
    
    public function delete_organization($id)
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->delete("{$apiBaseUrl}/organizations/{$id}");
    
        if ($response->successful()) {
            return response()->json(['success' => true, 'message' => 'Organization deleted successfully']);
        } else {
            // Extract error message from the API response, defaulting to a generic message
            $error = $response->json();
            $errorMessage = $error['message'] ?? 'Unable to delete Organization';
            
            return response()->json([
                'success' => false,
                'message' => $errorMessage
            ], $response->status());
        }
    }
    

    public function get_organization($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $response = Http::withToken($token)->get("{$apiBaseUrl}/organizations/get/{$id}");
    
          
    
            if ($response->successful()) {
                // Return the department data from the API response
                return response()->json($response->json()['organization']);
            } else {
                return response()->json(['status' => 404, 'message' => 'Organization not found.'], 404);
            }
        } catch (\Exception $e) {
            // Dump the exception message and die
            dd('An error occurred: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
}
