<?php

namespace App\Http\Controllers\SuperAdmin;

use CURLFile;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;
use App\DataTables\SuperAdmin\OrganizationDataTable;

class OrganizationController extends Controller
{
    private function fetch_users(): array
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->get("{$apiBaseUrl}/users?search=admin_normal");

        if ($response->successful()) {
            $users = $response->json()['users'] ?? [];
            // Extract only the necessary columns
            $users = array_map(function ($users) {
                return [
                    'id' => $users['id'],
                    'name' => $users['name'],
                ];
            }, $users);

            return $users;
        }

        return [];
    }
    public function organization_list()
    {
        $users = $this->fetch_users();
        return view('super_admin.organization.organization', compact('users'));
    }
   




    public function update_organization_form(Request $request, $id)
    {
        try {
            // Validate the incoming request
            $validatedData = $request->validate([
                'name' => 'required|string|max:255',
                'permissions' => 'required|array|min:1',
                'image' => 'nullable|file|mimes:jpeg,png,jpg,pdf,doc,docx|max:2048',
                'status' => 'required|string',
            ], [
                'name.required' => 'The name field is required',
                'permissions.required' => 'Please select at least one permission',
                'image.mimes' => 'The image must be a file of type: jpeg, png, jpg.',
                'image.max' => 'The image may not be greater than 2MB.',
            ]);

            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
          
            $requestBody = [
                'name' => $validatedData['name'],
                'permissions[]' => $validatedData['permissions'],
                'currency' => $request->currency,
                'status' => $validatedData['status'],
            ];

            // Handle the image file
            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $requestBody['image'] = new \CURLFile(
                    $image->getRealPath(),
                    $image->getMimeType(),
                    $image->getClientOriginalName()
                );
            }

           
            $ch = curl_init("{$apiBaseUrl}/organizations/update/{$id}");
            $apiBaseUrl = env('API_BASE_URL');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $token,
                'Content-Type: multipart/form-data',
            ]);

            // Use POST to send data
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $requestBody);

            // Execute the cURL request
            $response = curl_exec($ch);
            $responseCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

            if (curl_errno($ch)) {
                throw new \Exception(curl_error($ch));
            }

            curl_close($ch);

            $responseArray = json_decode($response, true);
            // dd($responseArray);
            if ($responseCode === 200) {
                return response()->json(['success' => true, 'message' => 'Organization updated successfully']);
            } else {
                if (isset($responseArray['errors'])) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Validation error',
                        'errors' => $responseArray['errors']
                    ], $responseCode);
                }

                return response()->json([
                    'success' => false,
                    'message' => $responseArray['message'] ?? 'Failed to add Organization.'
                ], $responseCode);
            }
        } catch (\Illuminate\Validation\ValidationException $e) {

            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }
    public function update_organization_status_form(Request $request, $id)
    {
        try {


            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $response = Http::withToken($token)->post("{$apiBaseUrl}/organizations/update/{$id}", [
                'status' => $request->status,
            ]);

            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Organization status updated  successfully']);
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update the organization.'
                ], $response->status());
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }

    public function delete_organization($id)
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->delete("{$apiBaseUrl}/organizations/{$id}");

        if ($response->successful()) {
            return response()->json(['success' => true, 'message' => 'Organization deleted successfully']);
        } else {
            // Extract error message from the API response, defaulting to a generic message
            $error = $response->json();
            $errorMessage = $error['message'] ?? 'Unable to delete Organization';

            return response()->json([
                'success' => false,
                'message' => $errorMessage
            ], $response->status());
        }
    }


    public function get_organization($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $response = Http::withToken($token)->get("{$apiBaseUrl}/organizations/get/{$id}");



            if ($response->successful()) {
                // Return the department data from the API response
                return response()->json($response->json()['organization']);
            } else {
                return response()->json(['status' => 404, 'message' => 'Organization not found.'], 404);
            }
        } catch (\Exception $e) {
            // Dump the exception message and die
            dd('An error occurred: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }


    //create organization step form 
    public function create_organization(){
        return view('super_admin.organization.create_organization');
    }
    public function add_organization_form(Request $request)
    {
        // Validate the incoming request data
        $request->validate([
            'user_name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255',
            'password' => 'required|string|min:8|confirmed',
            'permissions' => 'required|array|min:1',
            'organization_name' => 'required|string|max:255',
            'image' => 'nullable|file',
        ]);
    
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
    
            // Create User
            $userData = [
                'name' => $request->user_name,
                'email' => $request->email,
                'password' => $request->password,
                'password_confirmation' => $request->password_confirmation,
                'role' => 'organization',
                'status' => 'active',
            ];
            $userResponse = $this->curlRequest("{$apiBaseUrl}/users/store", $token, $userData);
    
            if ($userResponse['code'] !== 201) {
                return $this->handleApiError($userResponse, 'User creation failed');
            }
    
            $userId = $userResponse['body']['user']['id'];
    
            $orgData = [
                'name' => $request->organization_name,
                'currency' => $request->currency,
                'permissions[]' => $request->permissions,
                'status' => 'active',
                'user_id' => $userId,
            ];
    
            // Add image if present
            if ($request->hasFile('image')) {
                $orgData['image'] = new \CURLFile(
                    $request->file('image')->getPathname(),
                    $request->file('image')->getMimeType(),
                    $request->file('image')->getClientOriginalName()
                );
            }
    
            // Create Organization
            $orgResponse = $this->curlRequest("{$apiBaseUrl}/organizations/store", $token, $orgData);
    
            if ($orgResponse['code'] === 201) {
                return response()->json([
                    'success' => true,
                    'message' => 'Organization created successfully',
                    'data' => $orgResponse['body'],
                ], 201);
            }
    
            // If organization creation fails, delete the user
            $this->curlRequest("{$apiBaseUrl}/users/{$userId}", $token, [], 'DELETE');
    
            return $this->handleApiError($orgResponse, 'Organization creation failed');
    
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage(),
            ], 500);
        }
    }
    
    private function curlRequest($url, $token, $data, $method = 'POST')
    {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $token,
            'Accept: application/json',
        ]);
    
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        } elseif ($method === 'DELETE') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
        }
    
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
    
        return [
            'code' => $httpCode,
            'body' => json_decode($response, true),
        ];
    }
    
    private function handleApiError($response, $defaultMessage)
    {
        if ($response['code'] === 422 && isset($response['body']['errors'])) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $response['body']['errors'],
            ], 422);
        }
    
        return response()->json([
            'success' => false,
            'message' => $response['body']['message'] ?? $defaultMessage,
        ], $response['code']);
    }
    
    
}
