<?php

namespace App\Http\Controllers\Api;

use Exception;
use App\Models\Settings\Document;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use DataTables;

class DocumentController extends Controller
{

    public function get_documents_index()
{
    try {
        $org_id = Auth::user()->org_id;
            
        $documents = Document::where('org_id',$org_id)->get();


        return DataTables::of($documents)
            ->addColumn('count', function($row) {
                static $count = 0;
                return ++$count;
            })

            ->addColumn('download', function($row) {
                return '<a href="'. asset($row->document) .'" class="btn btn-success" download>Download</a>';
            })
            ->rawColumns(['download'])
            ->make(true);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Something went wrong & ' . $e->getMessage(),
        ]);
    }
}

    // Fetch all documents
    public function get_documents()
    {
        try {
            $org_id = Auth::user()->org_id;
            $documents = Document::where('org_id',$org_id)->get();

            return response()->json([
                "message" => "Documents fetched successfully",
                "documents" => $documents
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch documents.",
                "details" => $e->getMessage()
            ], 500);
        }
    }

    // Create a new document
    public function add_document(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'document' => 'required|file|max:4048',
                'expiry_date' => 'required',
                'remind_before' => 'required|in:7,15,30',
                'org_id' => 'required|exists:organizations,id',
                'employee_id' => 'required|exists:employee,id',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            // Handle file upload
            $documentName = Str::random(40) . '.' . $request->file('document')->getClientOriginalExtension();
            $documentPath = $request->file('document')->move(public_path('uploads/files/documents'), $documentName);

            $document = new Document();
            $document->name = $request->name;
            $document->org_id = $request->org_id;
            $document->employee_id = $request->employee_id;
            $document->document = 'uploads/files/documents/' . $documentName;
            $document->expiry_date = $request->expiry_date;
            $document->remind_before = $request->remind_before;
            $document->save();

            return response()->json([
                'success' => true,
                'message' => 'Document created successfully',
                'document' => $document
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create document.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    // Fetch a single document
    public function get_document($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $document = Document::where('org_id',$org_id)->findOrFail($id);

            return response()->json([
                'message' => 'Document retrieved successfully',
                'document' => $document
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Document not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve document.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    // Update a document
    public function update_document(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'document' => 'nullable|file|mimes:pdf,jpeg,png,jpg,doc,docx|max:2048',
                'expiry_date' => 'required',
                'remind_before' => 'required|in:7,15,30',
                'status' => 'required',
                'employee_id'=>'required|exists:employee,id',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $document = Document::findOrFail($id);

            // Handle the document file upload
            if ($request->hasFile('document')) {
                // Generate a unique file name
                $documentName = Str::random(40) . '.' . $request->file('document')->getClientOriginalExtension();
                $documentPath = public_path('uploads/files/documents');

                // Delete old document if it exists
                if ($document->document && file_exists(public_path($document->document))) {
                    unlink(public_path($document->document));
                }

                // Store the new document
                $request->file('document')->move($documentPath, $documentName);
                $document->document = 'uploads/files/documents/' . $documentName;
            }

            // Update remaining fields
            $document->name = $request->name;
            $document->status = $request->status;
            $document->expiry_date = $request->expiry_date;
            $document->remind_before = $request->remind_before;
            $document->employee_id = $request->employee_id;
            $document->save();

            return response()->json([
                'success' => true,
                'message' => 'Document updated successfully',
                'document' => $document
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Document not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update document',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    // Delete a document
    public function delete_document($id)
    {
        try {
            $document = Document::findOrFail($id);

            // Delete associated file
            if (file_exists(public_path($document->document))) {
                unlink(public_path($document->document));
            }

            $document->delete();

            return response()->json(['message' => 'Document deleted successfully']);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => 'Document not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'Failed to delete document',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
