<?php

namespace App\Http\Controllers\Api;

use Exception;
use App\Models\Settings\Employee;
use App\Models\Settings\TaxSlab;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Settings\Incomes;
use App\Models\Settings\Leave;
use App\Models\Settings\Loan;
use App\Models\Settings\Payment;
use App\Models\Settings\UserProfile;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use DataTables;
class EmployeeController extends Controller
{
    public function get_employees_index() {
        try {
            $org_id = Auth::user()->org_id;
    
            // Fetch employees with relationships
            $employees = Employee::where('org_id', $org_id)
                ->with([
                    'userProfile.user',
                    'salaryType',
                    'department',
                    'designation',
                    'allowances',
                    'shift',
                    'jobType',
                ])
                ->get();
    
            return DataTables::of($employees)
                ->addColumn('count', function ($row) {
                    static $count = 0;
                    return ++$count;
                })
                ->make(true);
        } catch (Exception $e) {
            return response()->json([
                "success" => false,
                "message" => "Failed to fetch employees.",
                "details" => $e->getMessage()
            ], 500);
        }
    }
    
    public function get_employees()
    {
        try {
            $org_id = Auth::user()->org_id;
            $employees = Employee::where('org_id',$org_id)->with([
                'userProfile.user',
                'salaryType',
                'department',
                'designation',
                'allowances',
                'shift',
                'jobType',
            ])->get();

            return response()->json([
                "message" => "Employees fetched successfully",
                "employees" => $employees
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch employees.",
                "details" => $e->getMessage()
            ], 500);
        }
    }

    public function add_employee(Request $request)
    {
        // Start a database transaction
        DB::beginTransaction();
    
        try {
            // Step 1: Validate the request data
            $validator = Validator::make($request->all(), [
                'org_id' => 'required',
                'joining_date' => 'required',
                'name' => 'required',
                'email' => [
                    'required',
                    'email',
                    'max:255',
                    function ($attribute, $value, $fail) use ($request) {
                        if ($request->filled('org_id')) {
                            $exists = \App\Models\User::where('email', $value)
                                ->where('org_id', $request->org_id)
                                ->exists();
                            if ($exists) {
                                $fail('The email has already been taken for the specified organization.');
                            }
                        }
                    },
                ],
                'password' => 'required|confirmed',
                'phone' => 'required',
                'address' => 'required',
                'cnic' => 'required',
                'gender' => 'required',
                'date_of_birth' => 'required',
                'salary_type_id' => 'required|exists:salary_types,id',
                'salary' => 'required',
                'total_leaves' => 'required',
                'department_id' => 'required|exists:departments,id',
                'designation_id' => 'required|exists:designations,id',
                'shift_id' => 'required|exists:shifts,id',
                'job_type_id' => 'required|exists:job_types,id',
                'image' => 'required|image|mimes:jpeg,png,jpg|max:2048', // Add image validation for profile
            ]);
    
            if ($validator->fails()) {
                throw new ValidationException($validator);
            }
    
            // Step 2: Create the User
            $userData = [
                'name' => $request->input('name'),
                'email' => $request->input('email'),
                'password' => bcrypt($request->input('password')),
                'role' => 'employee',
                'org_id' => $request->org_id,
            ];
    
            $user = User::create($userData);
    
            // Step 3: Handle the profile creation using the same structure as `add_profile`
            $imageName = Str::random(40) . '.' . $request->file('image')->getClientOriginalExtension();
            $imagePath = $request->file('image')->move(public_path('uploads/files/images'), $imageName);
    
            $profileData = [
                'user_id' => $user->id,
                'org_id' => $request->org_id,
                'phone' => $request->input('phone'),
                'address' => $request->input('address'),
                'cnic' => $request->input('cnic'),
                'gender' => $request->input('gender'),
                'date_of_birth' => $request->input('date_of_birth'),
                'image' => 'uploads/files/images/' . $imageName,
            ];
    
            $userProfile = UserProfile::create($profileData);
    
            // Step 4: Create the Employee
            $employee = new Employee();
            $employee->org_id = $request->org_id;
            $employee->joining_date = $request->joining_date;
            $employee->user_profile_id = $userProfile->id; // Use the created profile's ID
            $employee->salary_type_id = $request->salary_type_id;
            $employee->salary = $request->salary;
            $employee->total_leaves = $request->total_leaves;
            $employee->department_id = $request->department_id;
            $employee->designation_id = $request->designation_id;
            $employee->allowances_id = $request?->allowances_id;
            $employee->shift_id = $request->shift_id;
            $employee->job_type_id = $request->job_type_id;
            $employee->save();
    
            // Step 5: Commit the transaction (everything is successful)
            DB::commit();
    
            return response()->json([
                'success' => true,
                'message' => 'Employee created successfully',
                'employee' => $employee
            ], 201);
    
        } catch (ValidationException $e) {
            // Rollback the transaction on validation error
            DB::rollBack();
    
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
    
        } catch (Exception $e) {
            // Rollback the transaction on general exception
            DB::rollBack();
    
            return response()->json([
                'success' => false,
                'message' => 'Failed to create employee.',
                'details' => $e->getMessage()
            ], 500);
        }
    }
    

    public function get_employee($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $employee = Employee::where('org_id',$org_id)->with([
                'userProfile.user',
                'salaryType',
                'department',
                'designation',
                'allowances',
                'shift',
                'jobType',
                'workHistory',
                'education'
            ])->findOrFail($id);

            return response()->json([
                'message' => 'Employee retrieved successfully',
                'employee' => $employee
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Employee not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve employee.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function update_employee(Request $request, $id)
    {
        // Start a database transaction
        DB::beginTransaction();
    
        try {
            // Step 1: Validate the request data
            $validator = Validator::make($request->all(), [
                'joining_date' => 'required',
                'name' => 'required',
                'email' => [
                    'required',
                    'email',
                    'max:255',
                    function ($attribute, $value, $fail) use ($request, $id) {
                        if ($request->filled('org_id')) {
                            $exists = \App\Models\User::where('email', $value)
                                ->where('org_id', $request->org_id)
                                ->where('id', '!=', $id) // Exclude the current user by ID
                                ->exists();
                            if ($exists) {
                                $fail('The email has already been taken for the specified organization.');
                            }
                        }
                    },
                ],
                'phone' => 'required',
                'address' => 'required',
                'cnic' => 'required',
                'gender' => 'required',
                'date_of_birth' => 'required',
                'salary_type_id' => 'required|exists:salary_types,id',
                'salary' => 'required',
                'total_leaves' => 'required',
                'department_id' => 'required|exists:departments,id',
                'designation_id' => 'required|exists:designations,id',
                'shift_id' => 'required|exists:shifts,id',
                'job_type_id' => 'required|exists:job_types,id',
                'image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048', // Add image validation for profile
            ]);
    
            if ($validator->fails()) {
                throw new ValidationException($validator);
            }
    
            // Step 2: Fetch the existing User, Profile, and Employee by ID
            $user = User::findOrFail($id);
            $userProfile = UserProfile::where('user_id', $user->id)->firstOrFail();
            $employee = Employee::where('user_profile_id', $userProfile->id)->firstOrFail();
    
            // Step 3: Update the User
            $userData = [
                'name' => $request->input('name'),
                'email' => $request->input('email'),
            ];
    
            if ($request->filled('password')) {
                $userData['password'] = bcrypt($request->input('password'));
            }
    
            $user->update($userData);
    
            // Step 4: Handle profile image update if a new image is uploaded
            if ($request->hasFile('image')) {
                $imageName = Str::random(40) . '.' . $request->file('image')->getClientOriginalExtension();
                $imagePath = $request->file('image')->move(public_path('uploads/files/images'), $imageName);
    
                // Delete the old image (optional)
                if (File::exists(public_path($userProfile->image))) {
                    File::delete(public_path($userProfile->image));
                }
    
                $userProfile->image = 'uploads/files/images/' . $imageName;
            }
    
            // Step 5: Update the UserProfile
            $profileData = [
                'phone' => $request->input('phone'),
                'address' => $request->input('address'),
                'cnic' => $request->input('cnic'),
                'gender' => $request->input('gender'),
                'date_of_birth' => $request->input('date_of_birth'),
            ];
    
            $userProfile->update($profileData);
    
            // Step 6: Update the Employee
            $employeeData = [
                'org_id' => $request->org_id,
                'joining_date' => $request->joining_date,
                'salary_type_id' => $request->salary_type_id,
                'salary' => $request->salary,
                'total_leaves' => $request->total_leaves,
                'department_id' => $request->department_id,
                'designation_id' => $request->designation_id,
                'shift_id' => $request->shift_id,
                'job_type_id' => $request->job_type_id,
                'allowances_id' => $request->allowances_id,
            ];
    
            $employee->update($employeeData);
    
            // Step 7: Commit the transaction (everything is successful)
            DB::commit();
    
            return response()->json([
                'success' => true,
                'message' => 'Employee updated successfully',
                'employee' => $employee
            ], 200);
    
        } catch (ValidationException $e) {
            // Rollback the transaction on validation error
            DB::rollBack();
    
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
    
        } catch (Exception $e) {
            // Rollback the transaction on general exception
            DB::rollBack();
    
            return response()->json([
                'success' => false,
                'message' => 'Failed to update employee.',
                'details' => $e->getMessage()
            ], 500);
        }
    }
    

    public function delete_employee($id)
    {
        try {
            $employee = Employee::findOrFail($id);

            $employee->delete();

            return response()->json(['message' => 'Employee deleted successfully']);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => 'Employee not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'Failed to delete employee',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function get_payslip(Request $request, $id)
    {
        // try {
            // Validate the request for the month
            $request->validate([
                'month' => 'required|date_format:Y-m',
                'deductions'=>'nullable:string',
                'loans'=>'nullable:string',
                'tax'=>'nullable:string',
                'leave_deductions'=>'nullable:string',
                'salary_comment'=>'nullable:string',
            ]);

            $startMonth = $request->month . '-01'; // Start date of the given month
            $startMonthDate = new \DateTime($startMonth); // Convert to DateTime object
            $monthName = $startMonthDate->format('F');
            $org_id = Auth::user()->org_id;

            // Fetch employee details
            $employee = Employee::where('org_id', $org_id)
                ->with([
                    'userProfile.user',
                    'department',
                    'designation',
                    'shift',
                    'allowances',
                    'increments',
                    'deducations',
                    'loans',
                    'bonus',
                ])->findOrFail($id);

            // Check if the requested month is before the employee's joining date
            $joiningDate = new \DateTime($employee->joining_date);
            $nextMonthOfJoining = (clone $joiningDate)->format('Y-m');

            if ($request->month < $nextMonthOfJoining) {
                return response()->json([
                    'success' => false,
                    'message' => 'The requested month is not the next month after the employee\'s joining date.',
                ], 400);
            }


            $payment = Payment::where('employee_id', $employee->id)
            ->where('org_id', $org_id)
            ->where('salary_month', $startMonthDate->format('Y-m'))
            ->first();

            $paymentStatus = $payment ? 'paid' : 'unpaid';

            $requestedMonth = $startMonthDate->format('Y-m');

            if ($requestedMonth == $nextMonthOfJoining) {
                // Employee joined within the requested month, calculate days worked
                $joinDay = (int)$joiningDate->format('d'); // Get the day employee joined
                $daysInMonth = (int)$joiningDate->format('t'); // Get total days in the joining month
                $daysWorked = $daysInMonth - $joinDay + 1; // Calculate days worked from joining date
            } else {
                // Default days worked is 20 if not the joining month
                $daysWorked = 30;
            }
            $salary = $employee->salary ?? 0;
            $dailySalary = $salary / 30; 
            $salary = $dailySalary * $daysWorked;

            // Calculate increments for the given month and all previous months
            $incrementAmount = $employee->increments
                ->filter(function ($increment) use ($startMonthDate) {
                    $incrementDate = new \DateTime($increment->date);
                    // Include increments for the current month and all previous months
                    return $incrementDate->format('Y-m') <= $startMonthDate->format('Y-m');
                })
                ->map(function ($increment) use ($startMonthDate) {
                    $incrementDate = new \DateTime($increment->date);
                    // If the increment applies to the current month, prorate it, otherwise add the full amount
                    if ($incrementDate->format('Y-m') == $startMonthDate->format('Y-m')) {
                        $daysInIncrementMonth = 30; // Assume a 30-day month
                        $incrementDay = (int) $incrementDate->format('d');
                        $proratedDays = $daysInIncrementMonth - $incrementDay + 1;
                        return floor(($increment->amount / $daysInIncrementMonth) * $proratedDays);
                    }
                    // For previous months, add the entire increment amount
                    return $increment->amount;
                })
                ->sum();

                // Calculate increments for previous months up to the selected month (excluding current month if day hasn't fully passed)
                $preIncrementAmount = $employee->increments
                ->filter(function ($increment) use ($startMonthDate) {
                    $incrementDate = new \DateTime($increment->date);
                    return $incrementDate < $startMonthDate; // Include only increments from previous months
                })
                ->sum('amount');

            

            // Calculate allowances
            $allowanceAmount = 0;
            if (!is_null($employee->allowances)) {
                $allowanceDate = new \DateTime($employee->allowances['created_at']);
                if ($allowanceDate->format('Y-m') <= $startMonthDate->format('Y-m')) {
                    $allowanceAmount = (int) $employee->allowances['amount'];
                }
            }

            // Calculate deductions

            $deductionAmount = 0;
            if ($request->deductions === 'on') {
                $deductionAmount = $employee->deducations
                    ->filter(function ($deduction) use ($startMonthDate) {
                        $deductionDate = new \DateTime($deduction->date);
                        return $deductionDate->format('Y-m') == $startMonthDate->format('Y-m');
                    })
                    ->sum('amount');
            }

            // Calculate bonuses
            $bonusAmount = $employee->bonus
                ->filter(function ($bonus) use ($startMonthDate) {
                    $bonusDate = new \DateTime($bonus->date);
                    return $bonusDate->format('Y-m') == $startMonthDate->format('Y-m');
                })
                ->sum('amount');

            // Fetch the loan details
            $loan_deduction = 0;
            $loan_amount = 0;
            $loan_id = 0;
            if ($request->loans === 'on') {
                $loan = $employee->loans->first();
                if($loan){
                    $loan_amount = $loan ? $loan->amount : 0;
                    $installment = $loan ? $loan->installment : 1;
                    $loan_deduction = ($loan_amount > 0) ? floor($loan_amount / $installment) : 0;
                    $loan_id = $loan->id;
                }
                
            }

            // Fetch the unpaid leaves for the given month
            $leaveDeduction = 0;
            $unpaidLeaveDays = 0;
            if ($request->leave_deductions === 'on') {
                $unpaidLeaves = Leave::where('user_id', $employee->userProfile->user_id)
                    ->where('leave_status', 'unpaid')
                    ->where('org_id', $org_id)
                    ->where(function ($query) use ($startMonthDate) {
                        $query->whereMonth('start_date', $startMonthDate->format('m'))
                            ->whereYear('start_date', $startMonthDate->format('Y'));
                    })
                    ->get();

                $unpaidLeaveDays = 0;
                foreach ($unpaidLeaves as $leave) {
                    $startDate = new \DateTime($leave->start_date);
                    $endDate = new \DateTime($leave->end_date);
                    $diff = $endDate->diff($startDate)->days + 1;
                    $unpaidLeaveDays += $diff;
                }

                // Deduct salary for unpaid leaves (assuming 30-day month)
                $perDaySalary = $salary / 30;
                $leaveDeduction = $unpaidLeaveDays * $perDaySalary;
            }

            $netPay = ($salary + $incrementAmount) - ($leaveDeduction + $deductionAmount);

              // Calculate tax based on the tax slabs
            $tax_slabs = TaxSlab::where('org_id', $org_id)->orderBy('id', 'desc')->get();
            //dd($tax_slabs->toArray());

            $tax = 0;

            if ($request->tax === 'on') {
                $annualNetPay = $netPay * 12; 
                foreach ($tax_slabs as $slab) {
                    $slabLimit = (float)$slab->tax_slab;
                    $slabRate = (float)$slab->tax;       
                    $flatTax = (float)$slab->flat_tax; 
    
                    if ($annualNetPay > $slabLimit) {
                        $total_tax_amount = $annualNetPay - $slabLimit;
                        $taxableIncome = $total_tax_amount * ($slabRate / 100); 
                        $tax = $taxableIncome + $flatTax;
                        break;
                    }
                }
            }

            $new_salary = $salary + $preIncrementAmount;


            // Calculate net pay after tax
          // dd($annualNetPay, $slabLimit,$slabRate,$flatTax,$tax);
            $netPayAfterTax = (floor($netPay - ($tax / 12)) + $bonusAmount + $allowanceAmount) - $loan_deduction; // Adjust back to monthly net pay



            // Convert the net pay after tax to words
            $totalAmountInWords = $this->convert_number_to_words($netPayAfterTax);
            $total_earning = $new_salary + $bonusAmount + $incrementAmount - $preIncrementAmount + $allowanceAmount;
            $total_deduction = $deductionAmount + $loan_deduction + ($tax / 12) + $leaveDeduction;
            // Prepare the data to send to the frontend
            $payslipData = [
                'employee' => $employee,
                'basic_salary' => (int) $salary,
                'total_earning'=> (int) $total_earning,
                'total_deduction'=> (int) $total_deduction,
                'salary' => (int) $new_salary,
                'increments' => (int) $incrementAmount - $preIncrementAmount,
                'deductions' => (int) $deductionAmount,
                'allowances' => (int) $allowanceAmount,
                'loan_amount' => (int) $loan_amount,
                'loan_installment' => (int) $loan_deduction,
                'bonus_amount' => (int) $bonusAmount,
                'unpaid_leaves' => $unpaidLeaveDays,
                'leave_deduction' => (int) $leaveDeduction,
                'net_pay' => (int) $netPay,
                'tax' => (int) ($tax / 12) ,
                'net_pay_after_tax' => (int) $netPayAfterTax,
                'salary_month' => $monthName,
                'payment_status' => $paymentStatus, 
                'loan_id' => $loan_id, 
                'salary_comment' => $request?->salary_comment ?? "",
                'total_amount_in_words' => $totalAmountInWords,
            ];

            return response()->json([
                'success' => true,
                'message' => 'Payslip data fetched successfully',
                'data' => $payslipData,
            ], 200);
        // } catch (ModelNotFoundException $e) {
        //     return response()->json(['message' => 'Employee not found'], 404);
        // } catch (Exception $e) {
        //     return response()->json(['message' => 'Failed to fetch payslip data', 'details' => $e], 500);
        // }
    }

    public function get_payslip_based_on_days(Request $request, $id, $days)
    {
        try {
            // Validate the request for the month
            $request->validate([
                'month' => 'required|date_format:Y-m',
            ]);

            $startMonth = $request->month . '-01'; // Start date of the given month
            $startMonthDate = new \DateTime($startMonth); // Convert to DateTime object
            $monthName = $startMonthDate->format('F');
            $org_id = Auth::user()->org_id;

            // Fetch employee details
            $employee = Employee::where('org_id', $org_id)
                ->with([
                    'userProfile.user',
                    'department',
                    'designation',
                    'shift',
                    'allowances',
                    'increments',
                    'deducations',
                    'loans',
                    'bonus',
                ])->findOrFail($id);

            // Check if the requested month is before the employee's joining date
            $joiningDate = new \DateTime($employee->joining_date);
            $nextMonthOfJoining = (clone $joiningDate)->format('Y-m');

            if ($request->month < $nextMonthOfJoining) {
                return response()->json([
                    'success' => false,
                    'message' => 'The requested month is not the next month after the employee\'s joining date.',
                ], 400);
            }

            $payment = Payment::where('employee_id', $employee->id)
                ->where('org_id', $org_id)
                ->where('salary_month', $startMonthDate->format('Y-m'))
                ->first();

            $paymentStatus = $payment ? 'paid' : 'unpaid';

            // Salary calculation based on the number of days worked
            $salary = $employee->salary ?? 0;
            $dailySalary = $salary / 30; // Assume a 30-day month for calculation
            $totalSalary = $dailySalary * $days; // Salary based on days worked

            // Calculate increments for the given month and all previous months
            $incrementAmount = $employee->increments
                ->filter(function ($increment) use ($startMonthDate) {
                    $incrementDate = new \DateTime($increment->date);
                    return $incrementDate->format('Y-m') <= $startMonthDate->format('Y-m');
                })
                ->map(function ($increment) use ($startMonthDate) {
                    $incrementDate = new \DateTime($increment->date);
                    if ($incrementDate->format('Y-m') == $startMonthDate->format('Y-m')) {
                        $daysInIncrementMonth = 30; // Assume a 30-day month
                        $incrementDay = (int) $incrementDate->format('d');
                        $proratedDays = $daysInIncrementMonth - $incrementDay + 1;
                        return floor(($increment->amount / $daysInIncrementMonth) * $proratedDays);
                    }
                    return $increment->amount;
                })
                ->sum();

                $preIncrementAmount = $employee->increments
                ->filter(function ($increment) use ($startMonthDate) {
                    $incrementDate = new \DateTime($increment->date);
                    return $incrementDate < $startMonthDate; // Include only increments from previous months
                })
                ->sum('amount');


            // Calculate allowances
            $allowanceAmount = 0;
            if (!is_null($employee->allowances)) {
                $allowanceDate = new \DateTime($employee->allowances['created_at']);
                if ($allowanceDate->format('Y-m') <= $startMonthDate->format('Y-m')) {
                    $allowanceAmount = (int) $employee->allowances['amount'];
                }
            }

            // Calculate deductions
            $deductionAmount = $employee->deducations
                ->filter(function ($deduction) use ($startMonthDate) {
                    $deductionDate = new \DateTime($deduction->date);
                    return $deductionDate->format('Y-m') == $startMonthDate->format('Y-m');
                })
                ->sum('amount');

            // Calculate bonuses
            $bonusAmount = $employee->bonus
                ->filter(function ($bonus) use ($startMonthDate) {
                    $bonusDate = new \DateTime($bonus->date);
                    return $bonusDate->format('Y-m') == $startMonthDate->format('Y-m');
                })
                ->sum('amount');

            // Fetch the loan details
            $loan_deduction = 0;
            $loan_amount = 0;
            $loan_id = 0;
            $loan = $employee->loans->first();
            if($loan){
                $loan_amount = $loan ? $loan->amount : 0;
                $installment = $loan ? $loan->installment : 1;
                $loan_deduction = ($loan_amount > 0) ? floor($loan_amount / $installment) : 0;
                $loan_id = $loan->id;
            }

            // Fetch the unpaid leaves for the given month
            $unpaidLeaves = Leave::where('user_id', $employee->userProfile->user_id)
                ->where('leave_status', 'unpaid')
                ->where('org_id', $org_id)
                ->where(function ($query) use ($startMonthDate) {
                    $query->whereMonth('start_date', $startMonthDate->format('m'))
                        ->whereYear('start_date', $startMonthDate->format('Y'));
                })
                ->get();

            // Calculate the number of unpaid leave days
            $unpaidLeaveDays = 0;
            foreach ($unpaidLeaves as $leave) {
                $startDate = new \DateTime($leave->start_date);
                $endDate = new \DateTime($leave->end_date);
                $diff = $endDate->diff($startDate)->days + 1; // Include the end date
                $unpaidLeaveDays += $diff;
            }

            // Deduct salary for unpaid leaves
            $leaveDeduction = $unpaidLeaveDays * $dailySalary;
            $netPay = ($totalSalary + $incrementAmount) - ($leaveDeduction + $deductionAmount);

            // Get applicable tax slabs from the database
            $tax_slabs = TaxSlab::where('org_id', $org_id)->orderBy('id', 'desc')->get();
            //dd($tax_slabs->toArray());

            $tax = 0;
            
            // Annualize the net pay to apply the tax slabs correctly
            $annualNetPay = $netPay * 12; // Assuming this is a monthly salary


            // Calculate tax based on the tax slabs
            foreach ($tax_slabs as $slab) {
                $slabLimit = (float)$slab->tax_slab; // Tax slab limit
                $slabRate = (float)$slab->tax;       // Percentage tax rate for this slab
                $flatTax = (float)$slab->flat_tax;   // Flat tax amount for this slab

                // Check if income exceeds this slab limit
                if ($annualNetPay > $slabLimit) {
                    $total_tax_amount = $annualNetPay - $slabLimit;
                    $taxableIncome = $total_tax_amount * ($slabRate / 100); // Tax for the slab limit
                    $tax = $taxableIncome + $flatTax;
                    break;
                }
            }

            // Calculate net pay after tax
            // dd($annualNetPay, $slabLimit,$slabRate,$flatTax,$tax);
            $netPayAfterTax = (floor($netPay - ($tax / 12)) + $bonusAmount + $allowanceAmount) - $loan_deduction; // Adjust back to monthly net pay

            
            // Convert the net pay after tax to words
            $totalAmountInWords = $this->convert_number_to_words($netPayAfterTax);

            // Prepare the data to send to the frontend
            $payslipData = [
                'employee' => $employee,
                'basic_salary' => (int) $salary,
                'salary' => (int) $totalSalary + $preIncrementAmount,
                'increments' => (int) $incrementAmount - $preIncrementAmount,
                'deductions' => (int) $deductionAmount,
                'allowances' => (int) $allowanceAmount,
                'loan_amount' => (int) $loan_amount,
                'loan_installment' => (int) $loan_deduction,
                'bonus_amount' => (int) $bonusAmount,
                'unpaid_leaves' => $unpaidLeaveDays,
                'leave_deduction' => (int) $leaveDeduction,
                'net_pay' => (int) $netPay,
                'tax' => (int) ($tax / 12) ,
                'net_pay_after_tax' => (int) $netPayAfterTax,
                'salary_month' => $monthName,
                'loan_id' => $loan_id,
                'payment_status' => $paymentStatus,
                'total_amount_in_words' => $totalAmountInWords,
            ];

            return response()->json([
                'success' => true,
                'message' => 'Payslip data fetched successfully',
                'data' => $payslipData,
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([ 'success' => false,'message' => 'Employee not found'], 200);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch payslip data',
                'details' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ], 200);
        }
    }



    private function convert_number_to_words($number)
    {
        $hyphen = '-';
        $conjunction = ' and ';
        $separator = ', ';
        $negative = 'negative ';
        $decimal = ' point ';
        $dictionary = [
            0 => 'zero',
            1 => 'one',
            2 => 'two',
            3 => 'three',
            4 => 'four',
            5 => 'five',
            6 => 'six',
            7 => 'seven',
            8 => 'eight',
            9 => 'nine',
            10 => 'ten',
            11 => 'eleven',
            12 => 'twelve',
            13 => 'thirteen',
            14 => 'fourteen',
            15 => 'fifteen',
            16 => 'sixteen',
            17 => 'seventeen',
            18 => 'eighteen',
            19 => 'nineteen',
            20 => 'twenty',
            30 => 'thirty',
            40 => 'forty',
            50 => 'fifty',
            60 => 'sixty',
            70 => 'seventy',
            80 => 'eighty',
            90 => 'ninety',
            100 => 'hundred',
            1000 => 'thousand',
            1000000 => 'million',
            1000000000 => 'billion',
            1000000000000 => 'trillion',
        ];

        if (!is_numeric($number)) {
            return false;
        }

        if ($number < 0) {
            return $negative . $this->convert_number_to_words(abs($number));
        }

        $string = $fraction = null;

        if (strpos($number, '.') !== false) {
            list($number, $fraction) = explode('.', $number);
        }

        switch (true) {
            case $number < 21:
                $string = $dictionary[$number];
                break;
            case $number < 100:
                $tens = ((int) ($number / 10)) * 10;
                $units = $number % 10;
                $string = $dictionary[$tens];
                if ($units) {
                    $string .= $hyphen . $dictionary[$units];
                }
                break;
            case $number < 1000:
                $hundreds = $number / 100;
                $remainder = $number % 100;
                $string = $dictionary[$hundreds] . ' ' . $dictionary[100];
                if ($remainder) {
                    $string .= $conjunction . $this->convert_number_to_words($remainder);
                }
                break;
            default:
                $baseUnit = pow(1000, floor(log($number, 1000)));
                $numBaseUnits = (int) ($number / $baseUnit);
                $remainder = $number % $baseUnit;
                $string = $this->convert_number_to_words($numBaseUnits) . ' ' . $dictionary[$baseUnit];
                if ($remainder) {
                    $string .= $remainder < 100 ? $conjunction : $separator;
                    $string .= $this->convert_number_to_words($remainder);
                }
                break;
        }

        if (null !== $fraction && is_numeric($fraction)) {
            $string .= $decimal;
            $words = [];
            foreach (str_split((string) $fraction) as $number) {
                $words[] = $dictionary[$number];
            }
            $string .= implode(' ', $words);
        }

        return $string;
    }


    public function process_csv_and_store_payslips(Request $request)
{
    $errors = [];
    try {
        // Validate the request if needed
        $request->validate([
            'csv_file' => 'required|file',
        ]);

        // Open and parse the CSV file
        $csvFile = fopen($request->file('csv_file')->getPathname(), 'r');
        $headers = fgetcsv($csvFile); // Get the CSV headers
        
        while (($row = fgetcsv($csvFile)) !== false) {
            $data = array_combine($headers, $row);
            $email = trim($data['email']);
            $month = $data['month'];
            $daysWorked = $data['days'];
            $payslip_comment = $data['comment'] ?? "";

            // Find employee by email
            $user = User::select('id')->where('email', $email)->first();

            if (!$user) {
                $errors[] = "User with email $email not found.";
                continue; 
            }

            $userProfile = UserProfile::select('id')->where('user_id', $user->id)->first();
            
            if (!$userProfile) {
                $errors[] = "User profile for user $email not found.";
                continue; 
            }

            $employee = Employee::select('id', 'salary')->where('user_profile_id', $userProfile->id)->first();
            
            if (!$employee) {
                $errors[] = "Employee for user $email not found.";
                continue; 
            }

            // Check if payment already exists for this month and employee
            $existingPayment = Payment::where('employee_id', $employee->id)
                                      ->where('salary_month', $month)
                                      ->exists();

            if ($existingPayment) {
                $errors[] = "Payment already exists for employee $email for the month $month.";
                continue; // Skip if payment already exists for this month
            }

            // Create a fake Request object for get_payslip
            $fakeRequest = new Request(['month' => $month]);

            // Call the get_payslip_based_on_days method and get the payslip data
            $payslipResponse = $this->get_payslip_based_on_days($fakeRequest, $employee->id, $daysWorked);
            $payslipData = $payslipResponse->getData();

            // Check if payslip generation was successful
            if (isset($payslipData->success) && $payslipData->success == true) {
                // Store the payslip data in the Payment model
                Payment::create([
                    'org_id' => Auth::user()->org_id,
                    'employee_id' => $employee->id,
                    'salary' => $payslipData->data->salary,
                    'salary_comment' => $payslip_comment,
                    'allowances' => $payslipData->data->allowances,
                    'increments' => $payslipData->data->increments,
                    'deductions' => $payslipData->data->deductions,
                    'loan_amount' => $payslipData->data->loan_amount,
                    'loan_installment' => $payslipData->data->loan_installment,
                    'bonus_amount' => $payslipData->data->bonus_amount,
                    'unpaid_leaves' => $payslipData->data->unpaid_leaves,
                    'leave_deduction' => $payslipData->data->leave_deduction,
                    'net_pay' => $payslipData->data->net_pay,
                    'tax' => $payslipData->data->tax,
                    'net_pay_after_tax' => $payslipData->data->net_pay_after_tax,
                    'days' => $daysWorked,
                    'salary_month' => $month,
                ]);

                if ($payslipData->data->loan_id != 0) {
                    $remaining_amount = $payslipData->data->loan_amount - $payslipData->data->loan_installment;
                    $loan = Loan::find($payslipData->data->loan_id);
                    
                    if ($loan) {
                        $remaining_installment = $loan->installment - 1;
                        $status = $remaining_amount == 0 ? 'paid' : 'pending';
                        
                        $loan->update([
                            'report' => json_encode([
                                'loan_amount' => $payslipData->data->loan_amount,
                                'paid_amount' => $payslipData->data->loan_installment,
                                'remaining_amount' => $remaining_amount,
                                'remaining_installment' => $remaining_installment,
                                'pay_date' => Carbon::now(),
                            ]),
                            'status' => $status,
                        ]);
                    }
                }
            }else{
                $errors[] = $payslipData->message;
            }
        }

        fclose($csvFile);

        return response()->json([
            'message' => 'Payslips processed and stored successfully',
            'errors' => $errors
        ], 200);
    } catch (Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to fetch payslip data',
            'details' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ], 500);
    }
}

    

    public function get_report(Request $request)
{
    try {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'month' => 'nullable|date_format:Y-m',
            'year' => 'nullable|integer|between:2000,' . (date('Y') + 5),
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'type' => 'nullable|in:income,expense,both' // Optional
        ]);

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        $org_id = Auth::user()->org_id;

        // Initialize the query
        $query = Payment::with('employee.userProfile.user')->where('org_id', $org_id);

        // Handle month filter
        if ($request->filled('month')) {
            $month = $request->input('month');
            $query->where('salary_month', $month); // Filter by exact salary_month
        }

        // Handle year filter
        if ($request->filled('year')) {
            $year = $request->input('year');
            $query->where('salary_month', 'like', $year . '%'); // Filter by year (YYYY)
        }

        // Handle custom date range filter
        if ($request->filled('start_date') && $request->filled('end_date')) {
            $startDate = $request->input('start_date');
            $endDate = $request->input('end_date');
            // Convert to 'YYYY-MM' format
            $startMonth = (new \DateTime($startDate))->format('Y-m');
            $endMonth = (new \DateTime($endDate))->format('Y-m');
            $query->whereBetween('salary_month', [$startMonth, $endMonth]);
        }

        // Fetch the filtered data
        $payments = $query->get();

        // Map the data to include only the necessary fields
        $mappedPayments = $payments->map(function ($payment) {
            return [
                'payment_id' => $payment->id,
                'net_pay_after_tax' => $payment->net_pay_after_tax,
                'tax' => $payment->tax,
                'salary_month' => $payment->salary_month,
                'name' => $payment->employee->userProfile->user->name, // Access employee name through relations
            ];
        });

        $totalNetPay = $payments->sum('net_pay_after_tax');
        $totalIncrements = $payments->sum('increments');
        $totalDeductions = $payments->sum('deductions');

        // Return response
        return response()->json([
            'success' => true,
            'message' => 'Report fetched successfully',
            'report' => [
                'total_net_pay' => $totalNetPay,
                'total_increments' => $totalIncrements,
                'total_deductions' => $totalDeductions,
                'payments' => $mappedPayments
            ]
        ], 200);
    } catch (ValidationException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Validation error',
            'errors' => $e->errors()
        ], 422);
    } catch (Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to fetch report',
            'details' => $e->getMessage()
        ], 500);
    }
}

    
    public function get_payment($id){
        $org_id = Auth::user()->org_id;
        $query = Payment::with('employee',
        'employee.department', 
        'employee.designation',
        'employee.shift', 
        'employee.userProfile.user',
        )->where('org_id', $org_id)->where('id',$id)->get();
        return response()->json([
            'success' => true,
            'message' => 'Report fetched successfully',
            'payment' => $query
        ], 200);
    }


    public function income_report(Request $request)
    {
        try {
            // Validate the request
            $validator = Validator::make($request->all(), [
                'month' => 'nullable|date_format:Y-m',
                'year' => 'nullable|integer|between:2000,' . (date('Y') + 5),
                'start_date' => 'nullable|date',
                'end_date' => 'nullable|date|after_or_equal:start_date',
                'type' => 'nullable|in:income,expense,both'
            ]);
    
            if ($validator->fails()) {
                throw new ValidationException($validator);
            }
    
            $org_id = Auth::user()->org_id;
    
            // Initialize the query
            $query = Incomes::where('org_id', $org_id);
    
            // Handle month filter
            if ($request->filled('month')) {
                $month = $request->input('month');
                $startDate = $month . '-01';
                $endDate = date('Y-m-t', strtotime($startDate)); // Last day of the month
                $query->whereBetween('date', [$startDate, $endDate]);
            }
    
            // Handle year filter
            if ($request->filled('year')) {
                $year = $request->input('year');
                $startDate = $year . '-01-01';
                $endDate = $year . '-12-31'; // Full year range
                $query->whereBetween('date', [$startDate, $endDate]);
            }
    
            // Handle custom date range filter
            if ($request->filled('start_date') && $request->filled('end_date')) {
                $startDate = $request->input('start_date');
                $endDate = $request->input('end_date');
                $query->whereBetween('date', [$startDate, $endDate]);
            }
    
            // Determine the type filter
            $type = $request->input('type', 'income');
    
            
            $expenses = collect(); 
            $incomes = collect();
            if ($type == 'income') {
                $incomes = $query->where('type', 'income')->get();
            }else if ($type == 'expense') {
                $expenses = $query->where('type', 'expense')->get();
            } elseif ($type == 'both') {
                $expenses = Incomes::where('org_id', $org_id)
                    ->where('type', 'expense')
                    ->whereBetween('date', [$startDate ?? '2000-01-01', $endDate ?? date('Y-m-d')])
                    ->get();
                $incomes = Incomes::where('org_id', $org_id)
                    ->where('type', 'income')
                    ->whereBetween('date', [$startDate ?? '2000-01-01', $endDate ?? date('Y-m-d')])
                    ->get();
            }
    
            // Calculate totals
            $totalIncome = $incomes->sum('amount');
            $totalExpense = $expenses->sum('amount');
            $totalRevenue = $totalIncome - $totalExpense;
    
            return response()->json([
                'success' => true,
                'message' => 'Report fetched successfully',
                'report' => [
                    'total_income' => $totalIncome,
                    'total_expense' => $totalExpense,
                    'total_revenue' => $totalRevenue,
                    'type' => $type,
                    'incomes' => $incomes,
                    'expenses' => $expenses
                ]
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch report',
                'details' => $e->getMessage()
            ], 500);
        }
    }
    

    



}


