<?php

namespace App\Http\Controllers\Api;

use Exception;
use App\Models\Settings\Increment;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Auth;
use DataTables;
class IncrementController extends Controller
{
    public function get_increments_index()
    {
        try {
            $org_id = Auth::user()?->org_id;
            $increments = Increment::where('org_id', $org_id)
                ->with('employee.userProfile.user') // Assuming Increment model has a relationship with Employee
                ->get();
    
            return DataTables::of($increments)
                ->addColumn('count', function($row) {
                    static $count = 0;
                    return ++$count;
                })
                ->editColumn('employee_name', function($row) {
                    return $row->employee->userProfile->user->name ?? 'N/A';// Fetch employee name from related employee model
                })
                ->editColumn('amount', function($row) {
                    return number_format($row['amount'], 2); // Format amount
                })
                ->editColumn('date', function($row) {
                    return \Carbon\Carbon::parse($row['date'])->format('d-M-Y'); // Format date to d-M-Y format
                })
                ->editColumn('reason', function($row) {
                    return $row['reason'] ?? ''; // Limit reason string to 50 characters
                })
                ->editColumn('status', function($row) {
                    return $row['status'] ?? ''; 
                })
            
                ->rawColumns(['status'])
                ->make(true);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong & ' . $e->getMessage(),
            ]);
        }
    }
    
    public function get_increments()
    {
        try {
            $org_id = Auth::user()->org_id;
            $increments = Increment::where('org_id',$org_id)->with(['employee.userProfile.user'])
            ->get()
            ->map(function ($increment) {
                return [
                    'id' => $increment->id,
                    'employee_id' => $increment->employee_id,
                    'amount' => $increment->amount,
                    'date' => $increment->date,
                    'reason' => $increment->reason,
                    'status' => $increment->status,
                    'employee_name' => $increment->employee->userProfile->user->name ?? 'N/A',
                ];
            });

            return response()->json([
                "message" => "Increments fetched successfully",
                "increments" => $increments
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch increments.",
                "details" => $e->getMessage()
            ], 500);
        }
    }

    public function add_increment(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'employee_id' => 'required|exists:employee,id',
                'amount' => 'required|numeric',
                'date' => 'required|date',
                'reason' => 'required|string',
                'status' => 'required|string|in:active,inactive',
                'org_id' => 'required|exists:organizations,id',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $increment = new Increment();
            $increment->employee_id = $request->employee_id;
            $increment->amount = $request->amount;
            $increment->date = $request->date;
            $increment->org_id = $request->org_id;
            $increment->reason = $request->reason;
            $increment->status = $request->status;
            $increment->save();

            return response()->json([
                'success' => true,
                'message' => 'Increment created successfully',
                'increment' => $increment
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create increment.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function get_increment($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $increment = Increment::where('org_id',$org_id)->with(['employee.userProfile.user'])
            ->findOrFail($id);
            return response()->json([
                'message' => 'Increment retrieved successfully',
                'increment' => $increment
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Increment not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve increment.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function update_increment(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'employee_id' => 'required|exists:employee,id',
                'amount' => 'required|numeric',
                'date' => 'required|date',
                'reason' => 'required|string',
                'status' => 'required|string|in:active,inactive',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $increment = Increment::findOrFail($id);

            $increment->employee_id = $request->employee_id;
            $increment->amount = $request->amount;
            $increment->date = $request->date;
            $increment->reason = $request->reason;
            $increment->status = $request->status;
            $increment->save();

            return response()->json([
                'success' => true,
                'message' => 'Increment updated successfully',
                'increment' => $increment
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Increment not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update increment',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function delete_increment($id)
    {
        try {
            $increment = Increment::findOrFail($id);

            $increment->delete();

            return response()->json(['message' => 'Increment deleted successfully']);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => 'Increment not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'Failed to delete increment',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
