<?php

namespace App\Http\Controllers\Api;

use Exception;
use App\Models\Settings\PettyCash;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use DataTables;
use App\Models\Settings\Incomes;

class PettyCashController extends Controller
{
    // Get all petty cash records
    public function index(Request $request)
    {
        try {
            $employeeId = $request->input('employee_id');
            $pettyCashes = PettyCash::where('employee_id', $employeeId)->get();
    
            // Calculate total available income for petty cash
            $totalIncome = Incomes::where('employee_id', $employeeId)
                                  ->where('income_expence_type', 'petty_cash')
                                  ->sum('amount');
    
            // Calculate total used petty cash for the employee
            $totalUsed = PettyCash::where('employee_id', $employeeId)->sum('amount');
    
            // Calculate the remaining balance
            $balance = $totalIncome - $totalUsed;
            // Return the results as DataTables, with a success message
            return DataTables::of($pettyCashes)
                ->addColumn('count', function($row) {
                    static $count = 0;
                    return ++$count;
                })
                ->editColumn('date', function ($income) {
                    return \Carbon\Carbon::parse($income->date)->format('d-M-Y');
                })
                ->with([
                    'balance' => $balance 
                ])
                ->make(true);
    
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch Petty Cash. ' . $e->getMessage(),
            ], 500);
        }
        
    }

    // Add new petty cash record
    public function store(Request $request)
    {
        try {
            // Validate the request data
            $validator = Validator::make($request->all(), [
                'employee_id' => 'required|exists:employee,id',
                'amount' => 'required|numeric',
                'date' => 'required|date',
                'description' => 'nullable|string',
            ]);
    
            if ($validator->fails()) {
                throw new ValidationException($validator);
            }
    
            // Fetch all pending incomes for petty cash for the employee
            $totalAvailableIncome = Incomes::where('employee_id', $request->employee_id)
                                            ->where('income_expence_type', 'petty_cash')
                                            ->sum('amount');
    
            if ($totalAvailableIncome <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'No available petty cash income found for the specified employee.'
                ], 400);
            }
    
            // Calculate the total petty cash already used by the employee
            $totalUsedPettyCash = PettyCash::where('employee_id', $request->employee_id)->sum('amount');
    
            // Check if the new petty cash exceeds the available income
            $newTotal = $totalUsedPettyCash + $request->amount;
            if ($newTotal > $totalAvailableIncome) {
                return response()->json([
                    'success' => false,
                    'message' => 'The total petty cash exceeds the allowed limit.',
                    'available_amount' => $totalAvailableIncome,
                    'used_amount' => $totalUsedPettyCash
                ], 400);
            }
    
            // Create the new petty cash record
            $pettyCash = new PettyCash();
            $pettyCash->employee_id = $request->employee_id;
            $pettyCash->amount = $request->amount;
            $pettyCash->date = $request->date;
            $pettyCash->description = $request->description ?? '';
            $pettyCash->save();
    
            return response()->json([
                'success' => true,
                'message' => 'Petty cash record created successfully',
                'petty_cash' => $pettyCash
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create petty cash record',
                'details' => $e->getMessage()
            ], 500);
        }
    }
    

    // Update a specific petty cash record by ID
    public function update(Request $request, $id)
    {
        try {
            // Validate the request data
            $validator = Validator::make($request->all(), [
                'employee_id' => 'nullable|exists:employee,id',
                'amount' => 'nullable|numeric',
                'date' => 'nullable|date',
                'description' => 'nullable|string',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            // Fetch the petty cash record
            $pettyCash = PettyCash::findOrFail($id);

            // Fetch all pending incomes for petty cash for the employee
            $totalAvailableIncome = Incomes::where('employee_id', $pettyCash->employee_id)
                                            ->where('income_expence_type', 'petty_cash')
                                            ->sum('amount');

            if ($totalAvailableIncome <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'No available petty cash income found for the specified employee.'
                ], 400);
            }

            // If a new amount is provided, validate the total amount against the income
            if ($request->filled('amount')) {
                // Calculate the total used petty cash excluding the current record
                $totalUsedPettyCash = PettyCash::where('employee_id', $pettyCash->employee_id)
                                            ->where('id', '!=', $pettyCash->id)
                                            ->sum('amount');

                // Calculate the new total including the updated amount
                $newTotal = $totalUsedPettyCash + $request->amount;

                // Check if the new total exceeds the available income amount
                if ($newTotal > $totalAvailableIncome) {
                    return response()->json([
                        'success' => false,
                        'message' => 'The total petty cash exceeds the allowed limit.',
                        'available_amount' => $totalAvailableIncome,
                        'used_amount' => $totalUsedPettyCash
                    ], 400);
                }

                // Update the amount in the petty cash record
                $pettyCash->amount = $request->amount;
            }

            // Update other fields in the petty cash record
            $pettyCash->fill($request->only([
                'employee_id',
                'date',
                'description',
            ]));
            $pettyCash->save();
            
            return response()->json([
                'success' => true,
                'message' => 'Petty cash record updated successfully',
                'petty_cash' => $pettyCash
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Petty cash record not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update petty cash record',
                'details' => $e->getMessage()
            ], 500);
        }
    }
    


    

    // Get a specific petty cash record by ID
    public function show($id)
    {
        try {
            $pettyCash = PettyCash::findOrFail($id);

            return response()->json([
                'message' => 'Petty cash record retrieved successfully',
                'petty_cash' => $pettyCash
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Petty cash record not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve petty cash record',
                'details' => $e->getMessage()
            ], 500);
        }
    }

  
    
    
    

    // Delete a specific petty cash record by ID
    public function destroy($id)
    {
        try {
            $pettyCash = PettyCash::findOrFail($id);
            $pettyCash->delete();

            return response()->json([
                'success' => true,
                'message' => 'Petty cash record deleted successfully',
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Petty cash record not found',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete petty cash record',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
