<?php

namespace App\Http\Controllers\Api;

use Exception;
use Illuminate\Http\Request;
use App\Models\Settings\Shift;
use App\Models\Settings\Salary_Type;
use App\Models\Settings\JobType;
use App\Models\Settings\Allowance;
use App\Models\Settings\Education;
use App\Models\Settings\Department;
use App\Http\Controllers\Controller;
use App\Models\Settings\Designation;
use App\Models\Settings\WorkHistory;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use DataTables;

class SettingsController extends Controller
{

    //departments

    public function department_index()
    {
        try {
            $org_id = Auth::user()->org_id;
            $departments = Department::where('org_id', $org_id)->get();
            return DataTables::of($departments)
                ->addColumn('count', function ($row) {
                    static $count = 0;
                    return ++$count;
                })
                ->make(true);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong & ' . $e->getMessage(),
            ]);
        }
    }
    public function department_list()
    {
        try {
            $org_id = Auth::user()->org_id;
            $departments = Department::where('org_id', $org_id)->get();
            return response()->json([
                "message" => "Departments fetched successfully",
                "departments" => $departments
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch departments.",
                "details" => $e->getMessage()
            ], 500);
        }
    }
    public function get_department($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            // Fetch department with related designations
            $department = Department::with('designations')->where('org_id', $org_id)->findOrFail($id);

            return response()->json([
                'message' => 'Department retrieved successfully',
                'department' => $department
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Department not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve department.',
                'details' => $e->getMessage()
            ], 500);
        }
    }


    public function add_department_form(Request $request)
    {
        try {

            $request->validate([
                'org_id' => 'required|exists:organizations,id',
                'department' => [
                    'required',
                    function ($attribute, $value, $fail) use ($request) {
                        if ($request->filled('org_id')) {
                            $exists = Department::where('department', $value)
                                ->where('org_id', $request->org_id)
                                ->exists();
                            if ($exists) {
                                $fail('The department has already been taken for the specified organization.');
                            }
                        }
                    },
                ],
            ]);



            $department = new Department();
            $department->department = $request->department;
            $department->org_id = $request->org_id;
            $department->save();

            // Return success response
            return response()->json([
                'success' => true,
                'message' => 'Department created successfully',
                'department' => $department
            ], 201);
        } catch (ValidationException $e) {
            // Handle validation exceptions
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            // Handle other exceptions
            return response()->json([
                'success' => false,
                'message' => 'Failed to create department.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function update_department_form(Request $request, $id)
    {
        try {
            $request->validate([
                'department' => [
                    'required',
                    function ($attribute, $value, $fail) use ($request, $id) {
                        $exists = Department::where('department', $value)
                            ->where('org_id', Auth::user()->org_id)
                            ->where('id', '!=', $id) // Exclude the current department record
                            ->exists();
                        if ($exists) {
                            $fail('The department has already been taken for the specified organization.');
                        }
                    },
                ],
            ]);


            // Find the department by ID
            $department = Department::findOrFail($id); // Using findOrFail to handle invalid department ID
            $department->department = $request->department;
            $department->save();

            // Return success response
            return response()->json([
                'success' => true,
                'message' => 'Department updated successfully',
                'department' => $department
            ], 200); // Use status code 200 for successful updates
        } catch (ValidationException $e) {
            // Handle validation exceptions
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            // Handle case when the department is not found
            return response()->json([
                'success' => false,
                'message' => 'Department not found.',
            ], 404);
        } catch (Exception $e) {
            // Handle other exceptions
            return response()->json([
                'success' => false,
                'message' => 'Failed to update department.',
                'details' => $e->getMessage()
            ], 500);
        }
    }





    public function delete_department($id)
    {
        try {
            $department = Department::findOrFail($id);
            $department->delete();
            return response()->json([
                'message' => 'Department deleted successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Department not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to delete department.',
                'details' => $e->getMessage()
            ], 500);
        }
    }



    //designations









    public function designation_index()
    {
        try {
            $org_id = Auth::user()?->org_id;
            $designations = Designation::where('org_id', $org_id)
                ->with('department')
                ->get();

            return DataTables::of($designations)
                ->addColumn('count', function ($row) {
                    static $count = 0;
                    return ++$count;
                })
                ->editColumn('designation', function ($row) {
                    return $row['designation'];
                })
                ->editColumn('department_name', function ($row) {
                    return $row['department']['department'] ?? 'N/A';
                })
                ->make(true);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong & ' . $e->getMessage(),
            ]);
        }
    }


    public function designation_list()
    {
        try {
            $org_id = Auth::user()->org_id;
            $designations = Designation::with('department')->where('org_id', $org_id)->get();
            return response()->json([
                "message" => "Designations fetched successfully",
                "designations" => $designations
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch designations.",
                "details" => $e->getMessage()
            ], 500);
        }
    }
    public function get_designation($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $designation = Designation::with('department')->where('org_id', $org_id)->findOrFail($id);
            return response()->json([
                'message' => 'Designation retrieved successfully',
                'designation' => $designation
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Designation not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve Designation.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function add_designation_form(Request $request)
    {

        try {
            // Validate the request
            $request->validate([
                'org_id' => 'required|exists:organizations,id',
                'designation' => [
                    'required',
                    function ($attribute, $value, $fail) use ($request) {
                        if ($request->filled('org_id') && $request->filled('department_id')) {
                            $exists = Designation::where('designation', $value)
                                ->where('org_id', $request->org_id)
                                ->where('department_id', $request->department_id)
                                ->exists();
                            if ($exists) {
                                $fail('The designation has already been taken for the specified organization and department.');
                            }
                        }
                    },
                ],
                'department_id' => 'required|exists:departments,id',
            ]);


            $designation = new Designation();
            $designation->designation = $request->designation;
            $designation->department_id = $request->department_id;
            $designation->org_id = $request->org_id;
            $designation->save();
            // Return success response
            return response()->json([
                'success' => true,
                'message' => 'Designation created successfully',
                'department' => 'designation'
            ], 201);
        } catch (ValidationException $e) {
            // Handle validation exceptions
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            // Handle other exceptions
            return response()->json([
                'success' => false,
                'message' => 'Failed to create Designation.',
                'details' => $e->getMessage()
            ], 500);
        }
    }


    public function update_designation_form(Request $request, $id)
    {
        // Validate the request
        $request->validate([
            'designation' => [
                'required',
                function ($attribute, $value, $fail) use ($request, $id) {
                    $designationExists = Designation::where('designation', $value)
                        ->where('department_id', $request->department_id)
                        ->where('org_id', Auth::user()->org_id)
                        ->where('id', '!=', $id)
                        ->exists();

                    if ($designationExists) {
                        $fail('The designation has already been taken for the specified organization and department.');
                    }
                },
            ],
            'department_id' => 'required|exists:departments,id',
        ]);


        try {

            $designation = Designation::findOrFail($id);
            $designation->designation = $request->designation;
            $designation->department_id = $request->department_id;
            $designation->save();

            // Return success response
            return response()->json([
                'message' => 'Designation updated successfully',
                'designation' => $designation
            ], 200);
        } catch (ModelNotFoundException $e) {
            // Return error response for not found
            return response()->json([
                'error' => 'Designation not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            // Return error response for other failures
            return response()->json([
                'error' => 'Failed to update Designation.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function delete_designation($id)
    {
        try {

            $designation = Designation::findOrFail($id);
            $designation->delete();
            return response()->json([
                'message' => 'Designation deleted successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            // Return error response for not found
            return response()->json([
                'error' => 'Designation not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            // Return error response for other failures
            return response()->json([
                'error' => 'Failed to delete Designation.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

















    //allowances

    public function allowances_index()
    {
        try {
            $org_id = Auth::user()?->org_id;
            $allowances = Allowance::where('org_id', $org_id)->get();
    
            return DataTables::of($allowances)
                ->addColumn('count', function($row) {
                    static $count = 0;
                    return ++$count;
                })
                ->editColumn('name', function($row) {
                    return $row['name'];
                })
                ->editColumn('amount', function($row) {
                    return number_format($row['amount'], 2); // Format amount if necessary
                })
                ->make(true);
    
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong & ' . $e->getMessage(),
            ]);
        }
    }
    
    public function allowances_list()
    {
        try {
            $org_id = Auth::user()->org_id;
            $allowances = Allowance::where('org_id', $org_id)->get();
            return response()->json([
                "message" => "Allowances fetched successfully",
                "allowances" => $allowances
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch Allowances.",
                "details" => $e->getMessage()
            ], 500);
        }
    }
    public function get_allowances($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $allowance = Allowance::where('org_id', $org_id)->findOrFail($id);

            // Return success response with allowance data
            return response()->json([
                'message' => 'Allowance retrieved successfully',
                'allowance' => $allowance
            ], 200);
        } catch (ModelNotFoundException $e) {
            // Return error response for not found
            return response()->json([
                'error' => 'Allowance not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            // Return error response for other failures
            return response()->json([
                'error' => 'Failed to retrieve Allowance.',
                'details' => $e->getMessage()
            ], 500);
        }
    }


    public function add_allowances_form(Request $request)
    {
        try {

            $request->validate([
                'org_id' => 'required|exists:organizations,id',
                'amount' => 'required',
                'name' => [
                    'required',
                    function ($attribute, $value, $fail) use ($request) {
                        if ($request->filled('org_id')) {
                            $exists = Allowance::where('name', $value)
                                ->where('org_id', $request->org_id)
                                ->exists();
                            if ($exists) {
                                $fail('The allowance name has already been taken for the specified organization.');
                            }
                        }
                    },
                ],
            ]);

            $allowances = new Allowance();
            $allowances->name = $request->name;
            $allowances->amount = $request->amount;
            $allowances->org_id = $request->org_id;
            $allowances->save();

            // Return success response
            return response()->json([
                'success' => true,
                'message' => 'Allowance created successfully',
                'allowances' => $allowances
            ], 201);
        } catch (ValidationException $e) {
            // Handle validation exceptions
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            // Handle other exceptions
            return response()->json([
                'success' => false,
                'message' => 'Failed to create Allowance.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function update_allowances_form(Request $request, $id)
    {
        try {

            $request->validate([
                'amount' => 'required',
                'name' => [
                    'required',
                    function ($attribute, $value, $fail) use ($request, $id) {
                        $exists = Allowance::where('name', $value)
                            ->where('org_id', Auth::user()->org_id)
                            ->where('id', '!=', $id)
                            ->exists();
                        if ($exists) {
                            $fail('The allowance has already been taken for the specified organization.');
                        }
                    },
                ],
            ]);

            $allowances = Allowance::findOrFail($id);
            $allowances->name = $request->name;
            $allowances->amount = $request->amount;
            $allowances->status = $request?->status ?? $allowances->status;
            $allowances->save();

            // Return success response
            return response()->json([
                'success' => true,
                'message' => 'Allowance updated successfully',
                'allowances' => $allowances
            ], 200); // Use status code 200 for successful updates
        } catch (ValidationException $e) {
            // Handle validation exceptions
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            // Handle case when the department is not found
            return response()->json([
                'success' => false,
                'message' => 'Allowance not found.',
            ], 404);
        } catch (Exception $e) {
            // Handle other exceptions
            return response()->json([
                'success' => false,
                'message' => 'Failed to update Allowance.',
                'details' => $e->getMessage()
            ], 500);
        }
    }
    public function delete_allowances($id)
    {
        try {
            $allowances = Allowance::findOrFail($id);
            $allowances->delete();
            return response()->json([
                'message' => 'Allowance deleted successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Allowance not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to delete Allowance.',
                'details' => $e->getMessage()
            ], 500);
        }
    }












    //job type
    public function job_type_index()
    {
        try {
            $org_id = Auth::user()->org_id;
            $job_types = JobType::where('org_id', $org_id)->get();
            return DataTables::of($job_types)
                ->addColumn('count', function ($row) {
                    static $count = 0;
                    return ++$count;
                })
                ->make(true);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong & ' . $e->getMessage(),
            ]);
        }
    }
    public function job_type_list()
    {
        try {
            $org_id = Auth::user()->org_id;
            $job_types = JobType::where('org_id', $org_id)->get();
            return response()->json([
                "message" => "Job Types fetched successfully",
                "job_types" => $job_types
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch Job Types.",
                "details" => $e->getMessage()
            ], 500);
        }
    }
    public function get_job_type($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $job_type = JobType::where('org_id', $org_id)->findOrFail($id);

            // Return success response with job_type data
            return response()->json([
                'message' => 'Job Type retrieved successfully',
                'job_type' => $job_type
            ], 200);
        } catch (ModelNotFoundException $e) {
            // Return error response for not found
            return response()->json([
                'error' => 'Job Type not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            // Return error response for other failures
            return response()->json([
                'error' => 'Failed to retrieve Job Type.',
                'details' => $e->getMessage()
            ], 500);
        }
    }


    public function add_job_type_form(Request $request)
    {
        try {

            $request->validate([
                'org_id' => 'required|exists:organizations,id',
                'job_type' => [
                    'required',
                    function ($attribute, $value, $fail) use ($request) {
                        if ($request->filled('org_id')) {
                            $exists = JobType::where('job_type', $value)
                                ->where('org_id', $request->org_id)
                                ->exists();
                            if ($exists) {
                                $fail('The job type has already been taken for the specified organization.');
                            }
                        }
                    },
                ],
            ]);

            $job_types = new JobType();
            $job_types->job_type = $request->job_type;
            $job_types->org_id = $request->org_id;
            $job_types->save();

            // Return success response
            return response()->json([
                'success' => true,
                'message' => 'Job Type created successfully',
                'job_types' => $job_types
            ], 201);
        } catch (ValidationException $e) {
            // Handle validation exceptions
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            // Handle other exceptions
            return response()->json([
                'success' => false,
                'message' => 'Failed to create Job Type.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function update_job_type_form(Request $request, $id)
    {
        try {
            $request->validate([
                'job_type' => [
                    'required',
                    function ($attribute, $value, $fail) use ($request, $id) {
                        $exists = JobType::where('job_type', $value)
                            ->where('org_id', Auth::user()->org_id)
                            ->where('id', '!=', $id) // Exclude the current job type record
                            ->exists();
                        if ($exists) {
                            $fail('The job type has already been taken for the specified organization.');
                        }
                    },
                ],
            ]);

            $job_type = JobType::findOrFail($id);
            $job_type->job_type = $request->job_type;
            $job_type->save();

            // Return success response
            return response()->json([
                'success' => true,
                'message' => 'Job Type updated successfully',
                'job_type' => $job_type
            ], 200); // Use status code 200 for successful updates
        } catch (ValidationException $e) {
            // Handle validation exceptions
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            // Handle case when the department is not found
            return response()->json([
                'success' => false,
                'message' => 'Job Type not found.',
            ], 404);
        } catch (Exception $e) {
            // Handle other exceptions
            return response()->json([
                'success' => false,
                'message' => 'Failed to update Job Type.',
                'details' => $e->getMessage()
            ], 500);
        }
    }
    public function delete_job_type($id)
    {
        try {
            $job_type = JobType::findOrFail($id);
            $job_type->delete();
            return response()->json([
                'message' => 'Job Type deleted successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Job Type not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to delete Job Type.',
                'details' => $e->getMessage()
            ], 500);
        }
    }



    //work history

    public function work_history_list()
    {
        try {
            $org_id = Auth::user()->org_id;
            $work_history = WorkHistory::where('org_id', $org_id)->get();
            return response()->json([
                "message" => "Work History fetched successfully",
                "work_history" => $work_history
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch Work History.",
                "details" => $e->getMessage()
            ], 500);
        }
    }

    public function get_work_history($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $work_history = WorkHistory::where('org_id', $org_id)->findOrFail($id);

            return response()->json([
                'message' => 'Work History retrieved successfully',
                'work_history' => $work_history
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Work History not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve Work History.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function add_work_history_form(Request $request)
    {
        try {
            $request->validate([
                'org_id' => 'required|exists:organizations,id',
                'employee_id' => 'required|exists:employee,id',
                'company_name' => 'required|string|max:255',
                'designation' => 'required|string|max:255',
                'joining_date' => 'required',
                'leaving_date' => 'required|after_or_equal:joining_date',
                'reason' => 'required|string|max:255'
            ]);

            $work_history = new WorkHistory();
            $work_history->org_id = $request->org_id;
            $work_history->employee_id = $request->employee_id;
            $work_history->company_name = $request->company_name;
            $work_history->designation = $request->designation;
            $work_history->joining_date = $request->joining_date;
            $work_history->leaving_date = $request->leaving_date;
            $work_history->reason = $request->reason;
            $work_history->save();

            return response()->json([
                'success' => true,
                'message' => 'Work History created successfully',
                'work_history' => $work_history
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create Work History.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function update_work_history_form(Request $request, $id)
    {
        try {
            $request->validate([
                'company_name' => 'required|string|max:255',
                'designation' => 'required|string|max:255',
                'joining_date' => 'required',
                'leaving_date' => 'required|after_or_equal:joining_date',
                'reason' => 'required|string|max:255'
            ]);

            $work_history = WorkHistory::findOrFail($id);
            $work_history->company_name = $request->company_name;
            $work_history->designation = $request->designation;
            $work_history->joining_date = $request->joining_date;
            $work_history->leaving_date = $request->leaving_date;
            $work_history->reason = $request->reason;
            $work_history->save();

            return response()->json([
                'success' => true,
                'message' => 'Work History updated successfully',
                'work_history' => $work_history
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Work History not found.',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update Work History.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function delete_work_history($id)
    {
        try {
            $work_history = WorkHistory::findOrFail($id);
            $work_history->delete();
            return response()->json([
                'message' => 'Work History deleted successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Work History not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to delete Work History.',
                'details' => $e->getMessage()
            ], 500);
        }
    }













    //education



    public function education_list()
    {
        try {
            $org_id = Auth::user()->org_id;
            $educations = Education::where('org_id', $org_id)->get();
            return response()->json([
                "message" => "Education fetched successfully",
                "educations" => $educations
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch Education.",
                "details" => $e->getMessage()
            ], 500);
        }
    }

    public function get_education($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            $education = Education::where('org_id', $org_id)->findOrFail($id);
            return response()->json([
                'message' => 'Education retrieved successfully',
                'education' => $education
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Education not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to retrieve Education.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function add_education_form(Request $request)
    {
        try {
            $request->validate([
                'org_id' => 'required|exists:organizations,id',
                'employee_id' => 'required|exists:employee,id',
                'institute_name' => 'required|string|max:255',
                'last_degree' => 'required|string|max:255',
                'campus' => 'required|string|max:255',
                'session_start' => 'required|string|max:255',
                'session_end' => 'required|string|max:255',
                'attach_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png,doc,docx|max:2048', // Validate file
            ]);

            // Handle file upload
            if ($request->hasFile('attach_file')) {
                $file = $request->file('attach_file');
                $fileName = Str::random(40) . '.' . $file->getClientOriginalExtension();
                $filePath = $file->move(public_path('uploads/files/education'), $fileName);
            } else {
                $fileName = null; // Handle case if no file is uploaded
            }

            // Create new education record
            $education = new Education();
            $education->org_id = $request->org_id;
            $education->employee_id = $request->employee_id;
            $education->institute_name = $request->institute_name;
            $education->last_degree = $request->last_degree;
            $education->campus = $request->campus;
            $education->session_start = $request->session_start;
            $education->session_end = $request->session_end;
            $education->attach_file = $fileName;
            $education->save();

            return response()->json([
                'success' => true,
                'message' => 'Education created successfully',
                'education' => $education
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create Education.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function update_education_form(Request $request, $id)
    {
        try {
            $request->validate([
                'institute_name' => 'required|string|max:255',
                'last_degree' => 'required|string|max:255',
                'campus' => 'required|string|max:255',
                'session_start' => 'required|string|max:255',
                'session_end' => 'required|string|max:255',
                'attach_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png,doc,docx|max:2048', // Validate file
            ]);

            // Find existing education record
            $education = Education::findOrFail($id);

            // Handle file upload if a new file is provided
            if ($request->hasFile('attach_file')) {
                // Delete the old file if it exists
                if ($education->attach_file) {
                    $oldFilePath = public_path('uploads/files/education/' . $education->attach_file);
                    if (file_exists($oldFilePath)) {
                        unlink($oldFilePath);
                    }
                }

                $file = $request->file('attach_file');
                $fileName = Str::random(40) . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('uploads/files/education'), $fileName);
                $education->attach_file = $fileName;
            }

            // Update education record
            $education->institute_name = $request->institute_name;
            $education->last_degree = $request->last_degree;
            $education->campus = $request->campus;
            $education->session_start = $request->session_start;
            $education->session_end = $request->session_end;
            $education->save();

            return response()->json([
                'success' => true,
                'message' => 'Education updated successfully',
                'education' => $education
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Education not found.',
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update Education.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function delete_education($id)
    {
        try {
            $education = Education::findOrFail($id);

            // Delete the file if it exists
            if ($education->attach_file) {
                $filePath = public_path('uploads/files/education/' . $education->attach_file);
                if (file_exists($filePath)) {
                    unlink($filePath);
                }
            }

            $education->delete();

            return response()->json([
                'message' => 'Education deleted successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Education not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to delete Education.',
                'details' => $e->getMessage()
            ], 500);
        }
    }




    //shifts



    public function shift_list()
    {
        try {
            $org_id = Auth::user()->org_id;
            $shifts = Shift::where('org_id', $org_id)->get();
            return response()->json([
                "message" => "Shifts fetched successfully",
                "shifts" => $shifts
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch Shift.",
                "details" => $e->getMessage()
            ], 500);
        }
    }

    public function shift_index()
    {
        try {
            $org_id = Auth::user()->org_id;
            $shifts = Shift::where('org_id', $org_id)->get();
            return DataTables::of($shifts)
                ->addColumn('count', function ($row) {
                    static $count = 0;
                    return ++$count;
                })
                ->make(true);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong & ' . $e->getMessage(),
            ]);
        }
    }


    public function get_shift($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            // Find the department by ID or fail
            $shift = Shift::where('org_id', $org_id)->findOrFail($id);

            // Return success response with education data
            return response()->json([
                'message' => 'Shift retrieved successfully',
                'shift' => $shift
            ], 200);
        } catch (ModelNotFoundException $e) {
            // Return error response for not found
            return response()->json([
                'error' => 'Shift not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            // Return error response for other failures
            return response()->json([
                'error' => 'Failed to retrieve Shift.',
                'details' => $e->getMessage()
            ], 500);
        }
    }


    public function add_shift_form(Request $request)
    {
        try {

            $request->validate([
                'org_id' => 'required|exists:organizations,id',
                'shift' => [
                    'required',
                    function ($attribute, $value, $fail) use ($request) {
                        if ($request->filled('org_id')) {
                            $exists = Shift::where('shift', $value)
                                ->where('org_id', $request->org_id)
                                ->exists();
                            if ($exists) {
                                $fail('The shift has already been taken for the specified organization.');
                            }
                        }
                    },
                ],
            ]);


            $shift = new Shift();
            $shift->shift = $request->shift;
            $shift->org_id = $request->org_id;
            $shift->save();

            // Return success response
            return response()->json([
                'success' => true,
                'message' => 'Shift created successfully',
                'shift' => $shift
            ], 201);
        } catch (ValidationException $e) {
            // Handle validation exceptions
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            // Handle other exceptions
            return response()->json([
                'success' => false,
                'message' => 'Failed to create Shift.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function update_shift_form(Request $request, $id)
    {
        try {
            $request->validate([
                'shift' => [
                    'required',
                    function ($attribute, $value, $fail) use ($request, $id) {
                        $exists = Shift::where('shift', $value)
                            ->where('org_id', Auth::user()->org_id)
                            ->where('id', '!=', $id) // Exclude the current shift record
                            ->exists();
                        if ($exists) {
                            $fail('The shift has already been taken for the specified organization.');
                        }
                    },
                ],
            ]);

            $shift = Shift::findOrFail($id);
            $shift->shift = $request->shift;
            $shift->save();

            // Return success response
            return response()->json([
                'success' => true,
                'message' => 'Shift updated successfully',
                'shift' => $shift
            ], 200); // Use status code 200 for successful updates
        } catch (ValidationException $e) {
            // Handle validation exceptions
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            // Handle case when the department is not found
            return response()->json([
                'success' => false,
                'message' => 'Shift not found.',
            ], 404);
        } catch (Exception $e) {
            // Handle other exceptions
            return response()->json([
                'success' => false,
                'message' => 'Failed to update Shift',
                'details' => $e->getMessage()
            ], 500);
        }
    }
    public function delete_shift($id)
    {
        try {
            $shift = Shift::findOrFail($id);
            $shift->delete();
            return response()->json([
                'message' => 'Shift deleted successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Shift not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to delete Shift.',
                'details' => $e->getMessage()
            ], 500);
        }
    }



    //salary



    //   public function salary_list()
    //   {
    //       try {
    //           $org_id = Auth::user()->org_id;
    //           $salaries = Salary::where('org_id',$org_id)->get();
    //           return response()->json([
    //               "message" => "Salaries fetched successfully",
    //               "salaries" => $salaries
    //           ], 200);
    //       } catch (Exception $e) {
    //           return response()->json([
    //               "error" => "Failed to fetch Salaries.",
    //               "details" => $e->getMessage()
    //           ], 500);
    //       }
    //   }
    //   public function get_salary($id)
    //   {
    //       try {
    //           $org_id = Auth::user()->org_id;
    //           // Find the department by ID or fail
    //           $salary = Salary::where('org_id',$org_id)->findOrFail($id);

    //           // Return success response with education data
    //           return response()->json([
    //               'message' => 'Salary retrieved successfully',
    //               'salary' => $salary
    //           ], 200);
    //       } catch (ModelNotFoundException $e) {
    //           // Return error response for not found
    //           return response()->json([
    //               'error' => 'Salary not found.',
    //               'details' => $e->getMessage()
    //           ], 404);
    //       } catch (Exception $e) {
    //           // Return error response for other failures
    //           return response()->json([
    //               'error' => 'Failed to retrieve Salary.',
    //               'details' => $e->getMessage()
    //           ], 500);
    //       }
    //   }


    //   public function add_salary_form(Request $request)
    //   {
    //       try {

    //         $request->validate([
    //             'org_id' => 'required|exists:organizations,id',
    //             'salary' => [
    //                 'required',
    //                 function ($attribute, $value, $fail) use ($request) {
    //                     if ($request->filled('org_id')) {
    //                         $exists = Salary::where('salary', $value)
    //                             ->where('org_id', $request->org_id)
    //                             ->exists();
    //                         if ($exists) {
    //                             $fail('The salary has already been taken for the specified organization.');
    //                         }
    //                     }
    //                 },
    //             ],
    //         ]);


    //           $salary = new Salary();
    //           $salary->salary = $request->salary;
    //           $salary->org_id = $request->org_id;
    //           $salary->save();

    //           // Return success response
    //           return response()->json([
    //               'success' => true,
    //               'message' => 'Salary created successfully',
    //               'salary' => $salary
    //           ], 201);
    //       } catch (ValidationException $e) {
    //           // Handle validation exceptions
    //           return response()->json([
    //               'success' => false,
    //               'message' => 'Validation error',
    //               'errors' => $e->errors()
    //           ], 422);
    //       } catch (Exception $e) {
    //           // Handle other exceptions
    //           return response()->json([
    //               'success' => false,
    //               'message' => 'Failed to create Salary.',
    //               'details' => $e->getMessage()
    //           ], 500);
    //       }
    //   }

    //   public function update_salary_form(Request $request, $id)
    //   {
    //       try {
    //         $request->validate([
    //             'salary' => [
    //                 'required',
    //                 function ($attribute, $value, $fail) use ($request, $id) {
    //                     $exists = Salary::where('salary', $value)
    //                         ->where('org_id', Auth::user()->org_id)
    //                         ->where('id', '!=', $id) // Exclude the current salary record
    //                         ->exists();
    //                     if ($exists) {
    //                         $fail('The salary has already been taken for the specified organization.');
    //                     }
    //                 },
    //             ],
    //         ]);

    //           $salary = Salary::findOrFail($id);
    //           $salary->salary = $request->salary;
    //           $salary->save();

    //           // Return success response
    //           return response()->json([
    //               'success' => true,
    //               'message' => 'Salary updated successfully',
    //               'salary' => $salary
    //           ], 200); // Use status code 200 for successful updates
    //       } catch (ValidationException $e) {
    //           // Handle validation exceptions
    //           return response()->json([
    //               'success' => false,
    //               'message' => 'Validation error',
    //               'errors' => $e->errors()
    //           ], 422);
    //       } catch (ModelNotFoundException $e) {
    //           // Handle case when the department is not found
    //           return response()->json([
    //               'success' => false,
    //               'message' => 'Salary not found.',
    //           ], 404);
    //       } catch (Exception $e) {
    //           // Handle other exceptions
    //           return response()->json([
    //               'success' => false,
    //               'message' => 'Failed to update Salary',
    //               'details' => $e->getMessage()
    //           ], 500);
    //       }
    //   }
    //   public function delete_salary($id)
    //   {
    //       try {
    //           $salary = Salary::findOrFail($id);
    //           $salary->delete();
    //           return response()->json([
    //               'message' => 'Salary deleted successfully'
    //           ], 200);
    //       } catch (ModelNotFoundException $e) {
    //           return response()->json([
    //               'error' => 'Salary not found.',
    //               'details' => $e->getMessage()
    //           ], 404);
    //       } catch (Exception $e) {
    //           return response()->json([
    //               'error' => 'Failed to delete Salary.',
    //               'details' => $e->getMessage()
    //           ], 500);
    //       }
    //   }


    //salary_type


    public function salary_type_index()
    {
        try {
            $org_id = Auth::user()->org_id;
            $salary_types = Salary_Type::where('org_id', $org_id)->get();
            return DataTables::of($salary_types)
                ->addColumn('count', function ($row) {
                    static $count = 0;
                    return ++$count;
                })
                ->make(true);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong & ' . $e->getMessage(),
            ]);
        }
    }
    public function salary_type_list()
    {
        try {
            $org_id = Auth::user()->org_id;
            $salary_types = Salary_Type::where('org_id', $org_id)->get();
            return response()->json([
                "message" => "Salary Types fetched successfully",
                "salary_types" => $salary_types
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                "error" => "Failed to fetch Salary Types.",
                "details" => $e->getMessage()
            ], 500);
        }
    }
    public function get_salary_type($id)
    {
        try {
            $org_id = Auth::user()->org_id;
            // Find the department by ID or fail
            $salary_type = Salary_Type::where('org_id', $org_id)->findOrFail($id);

            // Return success response with education data
            return response()->json([
                'message' => 'Salary Type retrieved successfully',
                'salary_type' => $salary_type
            ], 200);
        } catch (ModelNotFoundException $e) {
            // Return error response for not found
            return response()->json([
                'error' => 'Salary Type not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            // Return error response for other failures
            return response()->json([
                'error' => 'Failed to retrieve Salary Type.',
                'details' => $e->getMessage()
            ], 500);
        }
    }


    public function add_salary_type_form(Request $request)
    {
        try {

            $request->validate([
                'org_id' => 'required|exists:organizations,id',
                'salary_type' => [
                    'required',
                    function ($attribute, $value, $fail) use ($request) {
                        if ($request->filled('org_id')) {
                            $exists = Salary_Type::where('salary_type', $value)
                                ->where('org_id', $request->org_id)
                                ->exists();
                            if ($exists) {
                                $fail('The salary type has already been taken for the specified organization.');
                            }
                        }
                    },
                ],
            ]);


            $salary_type = new Salary_Type();
            $salary_type->salary_type = $request->salary_type;
            $salary_type->org_id = $request->org_id;
            $salary_type->save();

            // Return success response
            return response()->json([
                'success' => true,
                'message' => 'Salary Type created successfully',
                'salary_type' => $salary_type
            ], 201);
        } catch (ValidationException $e) {
            // Handle validation exceptions
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            // Handle other exceptions
            return response()->json([
                'success' => false,
                'message' => 'Failed to create Salary Type.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function update_salary_type_form(Request $request, $id)
    {
        try {
            $request->validate([
                'salary_type' => [
                    'required',
                    function ($attribute, $value, $fail) use ($request, $id) {
                        $exists = Salary_Type::where('salary_type', $value)
                            ->where('org_id', Auth::user()->org_id)
                            ->where('id', '!=', $id) // Exclude the current salary type record
                            ->exists();
                        if ($exists) {
                            $fail('The salary type has already been taken for the specified organization.');
                        }
                    },
                ],
            ]);

            $salary_type = Salary_Type::findOrFail($id);
            $salary_type->salary_type = $request->salary_type;
            $salary_type->save();

            // Return success response
            return response()->json([
                'success' => true,
                'message' => 'Salary Type updated successfully',
                'salary_type' => $salary_type
            ], 200); // Use status code 200 for successful updates
        } catch (ValidationException $e) {
            // Handle validation exceptions
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (ModelNotFoundException $e) {
            // Handle case when the department is not found
            return response()->json([
                'success' => false,
                'message' => 'Salary Type not found.',
            ], 404);
        } catch (Exception $e) {
            // Handle other exceptions
            return response()->json([
                'success' => false,
                'message' => 'Failed to update Salary Type',
                'details' => $e->getMessage()
            ], 500);
        }
    }
    public function delete_salary_type($id)
    {
        try {
            $salary_type = Salary_Type::findOrFail($id);
            $salary_type->delete();
            return response()->json([
                'message' => 'Salary Type deleted successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'error' => 'Salary Type not found.',
                'details' => $e->getMessage()
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to delete Salary Type.',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
