<?php
namespace App\DataTables\SuperAdmin;

use Yajra\DataTables\DataTableAbstract;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;
use Illuminate\Support\Facades\Http;

class OrganizationDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param DataTableAbstract $query
     * @return DataTableAbstract
     */
    protected $imageBaseUrl;

    public function __construct()
    {
        $this->imageBaseUrl = env('API_URL_FOR_IMAGE');
    }
    public function dataTable($query): DataTableAbstract
    {
        return datatables()->of($query)
            ->addColumn('user_name', function ($item) {
                return $item['user']['name']; 
            })
            ->editColumn('image', function ($image) {
                if (empty($image['image'])) {
                    return 'No logo added';
                }
                $fullUrl = $this->imageBaseUrl . '/' . $image['image'];
                return '<img src="' . $fullUrl . '" alt="Document Image" style="width: 60px; height: 60px;">';
            })
            ->rawColumns(['image']); 
            
    }
    /**
     * Get the query source of dataTable.
     *
     * @return \Illuminate\Support\Collection
     */
    public function query()
    {
        // Fetch data from the external API
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->get("$apiBaseUrl/organizations");

        // Check if the response is successful
        if ($response->successful()) {
            // Return the JSON data from 'organizations' key
            return collect($response->json()['organizations']);
        }

        // Return an empty collection if the API call fails
        return collect([]);
    }

    /**
     * Optional method if you want to use the HTML builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
                    ->setTableId('organization_table')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    ->orderBy(1)
                    ->selectStyleSingle();
    }

    /**
     * Get the DataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('id'),
            Column::make('user_id'),
            Column::make('image')->title('Image'),
            Column::make('name'),
            Column::make('permissions'),
            Column::make('status'),
            Column::make('created_at'),
            Column::make('updated_at'),
            Column::make('user_name')->title('User Name') // Add this column
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Organization_' . date('YmdHis');
    }
}
