<?php

namespace App\Http\Controllers\Admin;

use League\Csv\Writer;
use SplTempFileObject;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;
use App\DataTables\EmployeeDataTable;
use Illuminate\Support\Facades\Session;


class EmployeeController extends Controller
{
    public function upload_csv(Request $request)
    {
        // Validate the request
        $request->validate([
            'csv_file' => 'required|file|mimes:csv,txt', // Adjust max size as needed
        ]);

        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        try {
            // Send the CSV file to the API
            $response = Http::withToken($token)->attach('csv_file', file_get_contents($request->file('csv_file')->getRealPath()), $request->file('csv_file')->getClientOriginalName())
                ->post("{$apiBaseUrl}/process/csv/payslips");

            // Check if response is successful
            if ($response->successful()) {
                return response()->json($response->json(), $response->status());
            } else {
                // Handle API error response
                return response()->json($response->json(), $response->status());
            }
        } catch (\Exception $e) {
            // Return any exceptions as JSON
            return response()->json(['message' => $e->getMessage()], 500);
        }
    }
    private function fetch_departments(): array
    {
        $org_id = session('org_id');
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->get("{$apiBaseUrl}/departments?org_id=$org_id");

        if ($response->successful()) {
            $departments = $response->json()['departments'] ?? [];
            // Extract only the necessary columns
            $departments = array_map(function ($departments) {
                return [
                    'id' => $departments['id'],
                    'department' => $departments['department'],
                ];
            }, $departments);

            return $departments;
        }

        return [];
    }
    private function fetch_designations(): array
    {
        $org_id = session('org_id');
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->get("{$apiBaseUrl}/designations?org_id=$org_id");

        if ($response->successful()) {
            $designations = $response->json()['designations'] ?? [];
            // Extract only the necessary columns
            $designations = array_map(function ($designations) {
                return [
                    'id' => $designations['id'],
                    'designation' => $designations['designation'],
                ];
            }, $designations);

            return $designations;
        }

        return [];
    }

    private function fetch_allowances(): array
    {
        $org_id = session('org_id');
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->get("{$apiBaseUrl}/allowances?org_id=$org_id");

        if ($response->successful()) {
            $allowances = $response->json()['allowances'] ?? [];
            // Extract only the necessary columns
            $allowances = array_map(function ($allowances) {
                return [
                    'id' => $allowances['id'],
                    'name' => $allowances['name'],
                ];
            }, $allowances);

            return $allowances;
        }

        return [];
    }
    private function fetch_job_types(): array
    {
        $org_id = session('org_id');
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->get("{$apiBaseUrl}/job_types?org_id=$org_id");

        if ($response->successful()) {
            $job_types = $response->json()['job_types'] ?? [];
            // Extract only the necessary columns
            $job_types = array_map(function ($job_types) {
                return [
                    'id' => $job_types['id'],
                    'job_type' => $job_types['job_type'],
                ];
            }, $job_types);

            return $job_types;
        }

        return [];
    }


    private function fetch_shift(): array
    {
        $org_id = session('org_id');
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->get("{$apiBaseUrl}/shift?org_id=$org_id");

        if ($response->successful()) {
            $shift = $response->json()['shifts'] ?? [];
            // Extract only the necessary columns
            $shift = array_map(function ($shift) {
                return [
                    'id' => $shift['id'],
                    'shift' => $shift['shift'],
                ];
            }, $shift);

            return $shift;
        }

        return [];
    }

    private function fetch_salary_type(): array
    {
        $org_id = session('org_id');
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->get("{$apiBaseUrl}/salary_type?org_id=$org_id");

        if ($response->successful()) {
            $salary_type = $response->json()['salary_types'] ?? [];
            // Extract only the necessary columns
            $salary_type = array_map(function ($salary_type) {
                return [
                    'id' => $salary_type['id'],
                    'salary_type' => $salary_type['salary_type'],
                ];
            }, $salary_type);

            return $salary_type;
        }

        return [];
    }
    public function create_employee()
    {
        $departments = $this->fetch_departments();
        $designations = $this->fetch_designations();

        $allowances = $this->fetch_allowances();
        $job_types = $this->fetch_job_types();

        $shifts = $this->fetch_shift();

        $salary_types = $this->fetch_salary_type();

        return view('admin.employee.create_employee', compact('departments', 'designations', 'allowances', 'job_types', 'shifts', 'salary_types'));
    }

    public function index()
    {
        return view('admin.employee.employee');
    }

    public function getEmployeeList(EmployeeDataTable $dataTable)
    {
        return $dataTable->ajax();
    }
    public function delete_employee($id)
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->delete("{$apiBaseUrl}/employees/{$id}");

        if ($response->successful()) {
            return response()->json(['success' => true, 'message' => 'Employee deleted successfully']);
        } else {
            // Ensure the error message is correctly fetched from the API response
            $error = $response->json();
            return response()->json([
                'success' => false,
                'message' => $error['message'] ?? 'Unable to delete Employee'
            ], $response->status());
        }
    }



    public function add_employee_form(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email',
            'password' => 'required|string|min:8|confirmed',
            'password_confirmation' => 'required_with:password|string|min:8',
            'phone' => 'required|string|max:15',
            'address' => 'nullable|string',
            'cnic' => 'required|string|max:15',
            'gender' => 'required|string',
            'date_of_birth' => 'required',
            'image' => 'required',
            'joining_date' => 'required',
            'salary_type_id' => 'required',
            'salary' => 'required',
            'department_id' => 'required',
            'designation_id' => 'required',
            'total_leaves' => 'required',
            'shift_id' => 'required',
            'job_type_id' => 'required',
            'image' => ''
        ], [
            'salary_type_id.required' => 'Please select at least one Salary Type',
            'department_id.required' => 'Please select a Department',
            'designation_id.required' => 'Please select a Designation',
            'shift_id.required' => 'Please select a Shift',
            'job_type_id.required' => 'Please select a Job Type',
        ]);


        $bearerToken = session('bearer_token');
        $orgId = session('org_id');
        $apiBaseUrl = env('API_BASE_URL');

        // Prepare the data for the first API call
        $employeeData = [
            'org_id' => $orgId,
            'name' => $request->input('name'),
            'email' => $request->input('email'),
            'password' => $request->input('password'),
            'password_confirmation' => $request->input('password_confirmation'),
            'role' => 'employee',
            'phone' => $request->input('phone'),
            'address' => $request->input('address'),
            'cnic' => $request->input('cnic'),
            'gender' => $request->input('gender'),
            'date_of_birth' => $request->input('date_of_birth'),
            'joining_date' => $request->input('joining_date'),
            'salary_type_id' => $request->input('salary_type_id'),
            'salary' => $request->input('salary'),
            'total_leaves' => $request->input('total_leaves'),
            'department_id' => $request->input('department_id'),
            'designation_id' => $request->input('designation_id'),
            'shift_id' => $request->input('shift_id'),
            'job_type_id' => $request->input('job_type_id'),
            'allowances_id' => $request->input('allowances_id', null),
        ];
        $ApiResponse = Http::withToken($bearerToken)
            ->attach('image', file_get_contents($request->file('image')->getRealPath()), $request->file('image')->getClientOriginalName())
            ->post("{$apiBaseUrl}/employees/store", $employeeData);

        // dd($ApiResponse->json());
        if ($ApiResponse->successful()) {

            $responseData = $ApiResponse->json();
            $employeeId = $responseData['employee']['id'] ?? null;

            if (Session::has('employee_id')) {
                Session::remove('employee_id');
            }
            if ($employeeId) {
                Session::put('employee_id', $employeeId);
            }
            return response()->json([
                'message' => 'Employee created successfully!',
                'employee' => $responseData
            ], 200);
        } else {
            return response()->json([
                'message' => 'Failed to add employee',
                'error' => $ApiResponse->json()
            ], $ApiResponse->status());
        }
    }
    public function add_employee_details_form(Request $request)
    {
        // Validate request inputs
        $request->validate([
            'institute_name' => 'required',
            'last_degree' => 'required',
            'campus' => 'required',
            'session_start' => 'required',
            'session_end' => 'required|after:session_start', // Ensure session_end is after session_start
            'company_name' => 'required',
            'designation' => 'required',
            'joining_date' => 'required', // Ensure joining_date is after session_start
            'leaving_date' => 'required|after:joining_date',  // Ensure leaving_date is after joining_date
            'reason' => 'required',
            'attach_file' => 'nullable|file' // Ensure attach_file is optional and a valid file
        ], [
            'session_end.after' => 'The session end date must be after the session start date.',
            'joining_date.after' => 'The joining date must be after the session start date.',
            'leaving_date.after' => 'The leaving date must be after the joining date.',
        ]);

        // Other code remains the same
        $bearerToken = session('bearer_token');
        $orgId = session('org_id');
        $employeeId = session('employee_id');
        $apiBaseUrl = env('API_BASE_URL');
        $educationData = [
            'employee_id' => $employeeId,
            'org_id' => $orgId,
            'institute_name' => $request->input('institute_name'),
            'last_degree' => $request->input('last_degree'),
            'campus' => $request->input('campus'),
            'session_start' => $request->input('session_start'),
            'session_end' => $request->input('session_end'),
        ];

        $educationRequest = Http::withToken($bearerToken);

        if ($request->hasFile('attach_file')) {
            $file = $request->file('attach_file');
            $educationRequest->attach('attach_file', file_get_contents($file->getRealPath()), $file->getClientOriginalName());
        }

        $secondApiResponse = $educationRequest->post("{$apiBaseUrl}/education/store", $educationData);
        // dd($secondApiResponse->json());
        $historyData = [
            'employee_id' => $employeeId,
            'org_id' => $orgId,
            'company_name' => $request->input('company_name'),
            'designation' => $request->input('designation'),
            'joining_date' => $request->input('joining_date'),
            'leaving_date' => $request->input('leaving_date'),
            'reason' => $request->input('reason'),
        ];

        $thirdApiResponse = Http::withToken($bearerToken)->post("{$apiBaseUrl}/work_history/store", $historyData);
        // dd($thirdApiResponse->json());
        if ($thirdApiResponse->successful()) {
            session()->forget('employee_id');

            return response()->json([
                'success' => 'Employee Detail added successfully!',
            ], 200);
        } else {
            return response()->json([
                'errors' => ['Failed to add Employee Details.'],
            ], 422);
        }
    }

    public function pay_salary(Request $request)
    {


        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $org_id = session('org_id');

            // No need to convert salary_month as it's already in the correct format
            // Prepare and send the API request
            $response = Http::withToken($token)->post("{$apiBaseUrl}/payments/store", [
                'org_id' => $org_id,
                'employee_id' => $request->employee_id,
                'increments' => $request->increments,
                'deductions' => $request->deductions,
                'loan_amount' => $request->loan_amount,
                'leave_deduction' => $request->leave_deduction,
                'unpaid_leaves' => $request->unpaid_leaves,
                'loan_installment' => $request->loan_installment,
                'bonus_amount' => $request->bonus_amount,
                'allowances' => $request->allowances,
                'salary' => $request->salary,
                'salary_comment' => $request->salary_comment,
                'net_pay' => $request->net_pay,
                'tax' => $request->tax,
                'loan_id' => $request->loan_id,
                'net_pay_after_tax' => $request->net_pay_after_tax,
                'salary_month' => $request->salary_month, // Use the salary_month directly
            ]);
            if ($response->successful()) {
                return redirect()->route('employees_index')->with('success', 'Payment Done Successfully');
            } else {
                // Handle non-successful response from the API
                $error = $response->json();
                return redirect()->back()->with('error', $error['error'] ?? 'Failed to process payment.');
            }
        } catch (\Exception $e) {
            // Catch any exceptions and return a detailed error message
            return redirect()->back()->with('error', 'Something went wrong: ' . $e->getMessage());
        }
    }
    public function update_employee($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');

            // Make the API call to get employee details
            $response = Http::withToken($token)->get("{$apiBaseUrl}/employees/{$id}");

            // Check if the response is successful
            if ($response->successful()) {
                $employee = $response->json()['employee']; // Adjust this key according to the API response structure

                $departments = $this->fetch_departments();
                $designations = $this->fetch_designations();
                $allowances = $this->fetch_allowances();
                $job_types = $this->fetch_job_types();
                $shifts = $this->fetch_shift();
                $salary_types = $this->fetch_salary_type();
                return view('admin.employee.update_employee', compact('employee', 'departments', 'designations', 'allowances', 'job_types', 'shifts', 'salary_types'));
            } else {
                return response()->json(['status' => 404, 'message' => 'Employee not found.'], 404);
            }
        } catch (\Exception $e) {
            // Log the exception message and return a proper error response
            Log::error('Error fetching employee: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred while fetching employee data. Please try again later.'], 500);
        }
    }
    public function update_employee_form(Request $request, $id)
    {
        // Validate input data
        $request->validate([
            'name' => 'required',
            'email' => 'required|email',
            'password' => 'nullable|string|min:8|confirmed',
            'password_confirmation' => 'nullable|string|min:8',
            'phone' => 'required|string|max:15',
            'address' => 'nullable|string',
            'cnic' => 'required|string|max:15',
            'gender' => 'required|string',
            'date_of_birth' => 'required|date',
            'image' => 'nullable|image',
            'joining_date' => 'required|date',
            'salary_type_id' => 'required',
            'salary' => 'required|numeric',
            'department_id' => 'required',
            'designation_id' => 'required',
            'total_leaves' => 'required|integer',
            'shift_id' => 'required',
            'job_type_id' => 'required',
        ], [
            'salary_type_id.required' => 'Please select at least one Salary Type',
            'department_id.required' => 'Please select a Department',
            'designation_id.required' => 'Please select a Designation',
            'shift_id.required' => 'Please select a Shift',
            'job_type_id.required' => 'Please select a Job Type',
        ]);

        $bearerToken = session('bearer_token');
        $org_id = session('org_id');
        $apiBaseUrl = env('API_BASE_URL');
        // Prepare employee data
        $employeeData = [
            'org_id' => $org_id,
            'name' => $request->input('name'),
            'email' => $request->input('email'),
            'role' => 'employee',
            'phone' => $request->input('phone'),
            'address' => $request->input('address'),
            'cnic' => $request->input('cnic'),
            'gender' => $request->input('gender'),
            'date_of_birth' => $request->input('date_of_birth'),
            'joining_date' => $request->input('joining_date'),
            'salary_type_id' => $request->input('salary_type_id'),
            'salary' => $request->input('salary'),
            'total_leaves' => $request->input('total_leaves'),
            'department_id' => $request->input('department_id'),
            'designation_id' => $request->input('designation_id'),
            'shift_id' => $request->input('shift_id'),
            'job_type_id' => $request->input('job_type_id'),
            'allowances_id' => $request->input('allowances_id', null),
        ];

        // Include the password only if it is provided
        if ($request->filled('password')) {
            $employeeData['password'] = $request->input('password');
            $employeeData['password_confirmation'] = $request->input('password_confirmation');
        }

        // Initialize cURL
        $curl = curl_init();

        // Set the URL and other options
        curl_setopt($curl, CURLOPT_URL, "{$apiBaseUrl}/employees/update/{$id}");
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_POST, true);

        // Set Authorization header
        $headers = [
            "Authorization: Bearer $bearerToken"
        ];

        // Handle image if it exists
        if ($request->hasFile('image')) {
            $employeeData['image'] = new \CURLFile($request->file('image')->getRealPath(), $request->file('image')->getMimeType(), $request->file('image')->getClientOriginalName());
        }

        // Set the POST fields and headers
        curl_setopt($curl, CURLOPT_POSTFIELDS, $employeeData);
        curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);

        // Execute the cURL request and get the response
        $response = curl_exec($curl);

        // Check for errors
        if (curl_errno($curl)) {
            // cURL error
            return response()->json([
                'message' => 'cURL Error: ' . curl_error($curl),
            ], 500);
        }

        // Get HTTP response code
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);

        // Close cURL
        curl_close($curl);

        // Decode the response (assuming it's JSON)
        $responseData = json_decode($response, true);
        if ($httpCode === 200) {
            return response()->json([
                'message' => 'Employee updated successfully!',
                'employee' => $responseData,
            ], 200);
        } else {
            // Handle failure
            return response()->json([
                'message' => 'Failed to update employee.',
                'error' => $responseData,
            ], $httpCode);
        }
    }


    public function generate_payslip(Request $request)
    {
        // Validate the incoming request data
        $validated = $request->validate([
            'employee_id' => 'required|numeric',
            'month' => 'required|date_format:Y-m',
        ]);

        // Extract validated data
        $employee_id = $validated['employee_id'];
        $month = $validated['month'];
        $apiBaseUrl = env('API_BASE_URL');
        // Build the API URL
        $api_url = "{$apiBaseUrl}/payslip/{$employee_id}?month={$month}";
        $token = session('bearer_token');

        // Make the API request
        $response = Http::withHeaders([
            'Authorization' => $token,
            'Accept' => 'application/json',
        ])->get($api_url);

        if ($response->successful()) {
            $data = $response->json();
            // Check payment status
            if ($data['data']['payment_status'] === 'paid') {
                return response()->json([
                    'success' => $data['message'],
                    'payment_status' => 'paid'
                ]);
            } else {
                // Payment status is unpaid, generate the salary slip
                $payload = [
                    'deductions' => $request->has('deductions') ? 'on' : 'off',
                    'loans' => $request->has('loans') ? 'on' : 'off',
                    'tax' => $request->has('tax') ? 'on' : 'off',
                    'leave_deductions' => $request->has('leave_deductions') ? 'on' : 'off',
                    'salary_comment' => $request->input('salary_comment', ''),
                    'month' => $validated['month'],
                ];

                // Generate salary slip with additional data
                $generate_response = Http::withHeaders([
                    'Authorization' => $token,
                    'Accept' => 'application/json',
                ])->get($api_url, $payload);

                if ($generate_response->successful()) {
                    $generated_data = $generate_response->json();
                    return view('admin.employee.salary_slip', [
                        'data' => $generated_data['data'],
                        'salary_date' => $month,
                    ]);
                } else {
                    return response()->json([
                        'message' => 'The requested month is not the next month after the employee joining date.',
                        'details' => $generate_response->json('message', 'Unknown error'),
                    ], $generate_response->status());
                }
            }
        } else {
            return response()->json([
                'message' => 'Failed to fetch payslip data',
                'details' => $response->json('message', 'Unknown error'),
            ], $response->status());
        }
    }
    public function fetch_paid_salary(Request $request)
    {

        $validated = $request->validate([
            'employee_id' => 'required|numeric',
            'month' => 'required|date_format:Y-m',
        ]);

        $employee_id = $validated['employee_id'];
        $month = $validated['month'];
        $apiBaseUrl = env('API_BASE_URL');
        // Build the correct API URL: /payslip/{employee_id}?month={month}
        $api_url = "{$apiBaseUrl}/payments/get/{$employee_id}?month={$month}";
        $token = session('bearer_token');
        if (!$token) {
            return redirect()->back()->with('error', 'Authentication token not found. Please login again.');
        }
        $response = Http::withHeaders([
            'Authorization' => $token,
            'Accept' => 'application/json',
        ])->get($api_url);
        // dd($response->json());
        if ($response->successful()) {
            $data = $response->json();
            return view('admin.employee.salary_slip', [
                'data' => $data['data'],
                'salary_month' => $month
            ]);
        } else {
            return redirect()->back()->with('error', 'Failed to fetch payslip data');
        }
    }

    public function generate_unpaid_payslip(Request $request)
    {
        // Validate the incoming request data
        $validated = $request->validate([
            'employee_id' => 'required|numeric',
            'month' => 'required|date_format:Y-m',
            'deductions' => 'nullable',
            'loans' => 'nullable',
            'tax' => 'nullable',
            'leave_deductions' => 'nullable',
            'salary_comment' => 'nullable|string|max:255',
        ]);

        // Extract validated data
        $employee_id = $validated['employee_id'];
        $month = $validated['month'];
        $apiBaseUrl = env('API_BASE_URL');
        $api_url = "{$apiBaseUrl}/payslip/{$employee_id}?month={$month}";
        $token = session('bearer_token');

        // Prepare the payload for the API request
        $payload = [
            'deductions' => $request->deductions,
            'loans' => $request->loans,
            'tax' => $request->tax,
            'leave_deductions' => $request->leave_deductions,
            'salary_comment' => $validated['salary_comment'],
            'month' => $validated['month'],
        ];

        $response = Http::withHeaders([
            'Authorization' => $token,
            'Accept' => 'application/json',
        ])->get($api_url, $payload);

        if ($response->successful()) {
            $data = $response->json();
            return view('admin.employee.salary_slip', [
                'data' => $data['data'],
                'salary_date' => $month
            ]);
        } else {
            // Output response details for debugging
            return response()->json([
                'message' => 'Failed to fetch payslip data',
                'details' => $response->json('message', 'Unknown error'),
            ], $response->status());
        }
    }

    public function export_employee_csv() {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
    
        // Call the API to get employee data
        $response = Http::withToken($token)->get("{$apiBaseUrl}/employees");
    
        // Check if the response is successful
        if ($response->successful()) {
            $employees = $response->json()['employees'];
    
            // Create a new CSV writer instance
            $csv = Writer::createFromFileObject(new SplTempFileObject());
            
            // Insert the header
            $csv->insertOne(['Name', 'Email', 'Department', 'Designation', 'Phone', 'Address','Joining Date', 'Salary']);
            
            // Insert employee data
            foreach ($employees as $employee) {
                $userProfile = $employee['user_profile']['user'];
                $department = $employee['department']['department'];
                $designation = $employee['designation']['designation'];
    
                $csv->insertOne([
                    $userProfile['name'],
                    $userProfile['email'],
                    $department,
                    $designation,
                    $employee['user_profile']['phone'],
                    $employee['user_profile']['address'],
                    $employee['joining_date'],
                    $employee['salary'],
                ]);
            }
    
            // Create a response for download
            return response((string) $csv)
                ->header('Content-Type', 'text/csv')
                ->header('Content-Disposition', 'attachment; filename="employees.csv"');
        }
    
        // Handle error response
        return response()->json(['error' => 'Unable to fetch employee data'], Response::HTTP_BAD_REQUEST);
    }
}
