<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;

class ProfileController extends Controller
{
    private function fetch_organization(): array
    {
        $org_id = session('org_id'); // Get org_id from session
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');

        // Send a GET request to the API to fetch organization data
        $response = Http::withToken($token)->get("{$apiBaseUrl}/organizations/get/{$org_id}");

        if ($response->successful()) {
            $organization = $response->json()['organization'] ?? [];

            // Extract only the necessary columns
            return [
                'id' => $organization['id'],
                'image' => $organization['image'],
                'name' => $organization['name'],
            ];
        }

        return [];
    }


    public function profile()
    {
        $organization = $this->fetch_organization(); // Fetch organization data
        return view('admin.profile.profile', compact('organization')); // Pass data to view
    }

    public function update_profile_password_form(Request $request, $id)
    {

        $request->validate([
            'password' => 'required|string|min:8|confirmed',
        ], [
            'password.required' => 'The password field is required.',
            'password.min' => 'The password must be at least 8 characters.',
            'password.confirmed' => 'The password confirmation does not match.',
        ]);

        try {

            $token = session('bearer_token');

            $apiBaseUrl = env('API_BASE_URL');

            $response = Http::withToken($token)->put("{$apiBaseUrl}/users/update/{$id}", [
                'password' => $request->password,
                'password_confirmation' => $request->password_confirmation,
            ]);
            // dd($response->json());

            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Password updated successfully']);
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update user.'
                ], $response->status());
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }
    public function update_organization_logo(Request $request, $id)
    {
        try {
            // Validate the incoming request
            $validatedData = $request->validate([
                'image' => 'required|file|mimes:jpeg,png,jpg,pdf,doc,docx|max:2048',
            ], [
                'image.required' => 'Please upload a logo image',
                'image.mimes' => 'The image must be a file of type: jpeg, png, jpg, pdf, doc, docx.',
                'image.max' => 'The image may not be greater than 2MB.',
            ]);

            // Retrieve the bearer token from the session
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            // Make an HTTP request to update the logo
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->attach(
                'image',
                file_get_contents($request->file('image')->getRealPath()),
                $request->file('image')->getClientOriginalName()
            )->post("$apiBaseUrl/organizations/update/{$id}");

            $apiBaseUrl = env('API_URL_FOR_IMAGE');
            $updatedImagePath  = $response->json('organization.image');
            $updatedImageUrl = rtrim($apiBaseUrl, '/') . '/' . ltrim($updatedImagePath, '/');
            session(['logo' => $updatedImageUrl]);


            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Organization logo updated successfully']);
            } else {

                return response()->json([

                    'success' => false,
                    'message' => $response->json('message') ?? 'Failed to update the logo.'
                ], $response->status());
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Handle validation errors
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            // Handle general exceptions
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }


    public function employee_profile()
    {
        $employee_id = session('employee_id');
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');

        // Send a GET request to the API to fetch employee data
        $response = Http::withToken($token)->get("{$apiBaseUrl}/employees/{$employee_id}");

        // Check if the request was successful
        if ($response->successful()) {
            $employee = $response->json('employee'); // Get the 'employee' data from the response
            return view('admin.profile.employee_profile', compact('employee'));
        } else {
            // Handle the error if the API request fails
            return redirect()->back()->with('error', 'Failed to retrieve employee data.');
        }
    }

    public function add_education_form(Request $request)
    {
        $request->validate([
            'institute_name' => 'required',
            'last_degree' => 'required',
            'campus' => 'required',
            'session_start' => 'required|date',
            'session_end' => 'required|date|after:session_start',
            'attach_file' => 'nullable|file|mimes:png,jpg,jpeg,svg,gif|max:2048', // Include specific file types and max size
        ], [
            'session_end.after' => 'The session end date must be after the session start date.',
        ]);

        // Get bearer token and other session data
        $bearerToken = session('bearer_token');
        $orgId = session('org_id');
        $employeeId = session('employee_id');
        $apiBaseUrl = env('API_BASE_URL');
        // Prepare the education data
        $educationData = [
            'employee_id' => $employeeId,
            'org_id' => $orgId,
            'institute_name' => $request->input('institute_name'),
            'last_degree' => $request->input('last_degree'),
            'campus' => $request->input('campus'),
            'session_start' => $request->input('session_start'),
            'session_end' => $request->input('session_end'),
        ];

        // Initialize cURL session
        $ch = curl_init();

        // Prepare the file for the cURL request if attached
        if ($request->hasFile('attach_file')) {
            $file = $request->file('attach_file');
            $filePath = $file->getRealPath();
            $fileName = $file->getClientOriginalName();

            $cFile = new \CURLFile($filePath, $file->getMimeType(), $fileName);
            $educationData['attach_file'] = $cFile;
        }

        // Set cURL options
        curl_setopt($ch, CURLOPT_URL, "$apiBaseUrl/education/store");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $bearerToken,
            'Accept: application/json'
        ]);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $educationData);

        // Execute the cURL request
        $response = curl_exec($ch);

        // Handle errors
        if (curl_errno($ch)) {
            return response()->json([
                'errors' => ['Failed to add education details: ' . curl_error($ch)],
            ], 422);
        }

        // Close the cURL session
        curl_close($ch);

        // Decode the response
        $apiResponse = json_decode($response, true);

        // Check if the API request was successful
        if (isset($apiResponse['success'])) {
            return redirect()->back()->with('success', 'Education added successfully');
        } else {
            return response()->json([
                'errors' => ['Failed to add education details.'],
            ], 422);
        }
    }

    public function update_education_form(Request $request, $id)
    {
        // dd($request->all());
        $request->validate([
            'institute_name' => 'required',
            'last_degree' => 'required',
            'campus' => 'required',
            'session_start' => 'required|date',
            'session_end' => 'required|date|after:session_start',
            'attach_file' => 'nullable|file|mimes:png,jpg,jpeg,svg,gif|max:2048' // Optional file validation
        ], [
            'session_end.after' => 'The session end date must be after the session start date.',
        ]);

        // Get bearer token and other session data
        $bearerToken = session('bearer_token');
        $orgId = session('org_id');
        $employeeId = session('employee_id');
        $apiBaseUrl = env('API_BASE_URL');
        $educationData = [
            'employee_id' => $employeeId,
            'org_id' => $orgId,
            'institute_name' => $request->input('institute_name'),
            'last_degree' => $request->input('last_degree'),
            'campus' => $request->input('campus'),
            'session_start' => $request->input('session_start'),
            'session_end' => $request->input('session_end'),
        ];

        // Initialize cURL session
        $ch = curl_init();

        // Prepare the file for the cURL request if attached
        if ($request->hasFile('attach_file')) {
            $file = $request->file('attach_file');
            $filePath = $file->getRealPath();
            $fileName = $file->getClientOriginalName();

            $cFile = new \CURLFile($filePath, $file->getMimeType(), $fileName);
            $educationData['attach_file'] = $cFile;
        }

        // Set cURL options
        curl_setopt($ch, CURLOPT_URL, "$apiBaseUrl/education/update/" . $id);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST'); // Use PUT for update
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $bearerToken,
            'Accept: application/json'
        ]);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $educationData);

        // Execute the cURL request
        $response = curl_exec($ch);

        // Handle errors
        if (curl_errno($ch)) {
            return response()->json([
                'errors' => ['Failed to update education details: ' . curl_error($ch)],
            ], 422);
        }

        curl_close($ch);


        $apiResponse = json_decode($response, true);
        // dd($apiResponse);
        if (isset($apiResponse['success'])) {
            return redirect()->back()->with('success', 'Education update successfully');
        } else {
            return response()->json([
                'errors' => ['Failed to update education details.'],
            ], 422);
        }
    }
    public function add_work_history(Request $request)
    {
        $request->validate([
            'company_name' => 'required',
            'designation' => 'required',
            'joining_date' => 'required|date',
            'leaving_date' => 'required|date|after:session_start',
            'reason' => 'required',
        ]);

        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $org_id = session('org_id');
            $employeeId = session('employee_id');
            // Prepare the request by attaching the file and sending other data as form fields
            $response = Http::withToken($token)->post("{$apiBaseUrl}/work_history/store", [
                'employee_id' => $employeeId,
                'org_id' => $org_id,
                'company_name' => $request->company_name,
                'designation' => $request->designation,
                'joining_date' => $request->joining_date,
                'leaving_date' => $request->leaving_date,
                'reason' => $request->reason,
            ]);
            // dd($response->json());

            if ($response->successful()) {
                return redirect()->back()->with('success', 'Experience update successfully');
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to add Work History '
                ], $response->status());
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }
    public function update_work_history(Request $request, $id)
    {
        $request->validate([
            'company_name' => 'required',
            'designation' => 'required',
            'joining_date' => 'required|date',
            'leaving_date' => 'required|date|after:joining_date',
            'reason' => 'required',
        ]);

        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $org_id = session('org_id');
            $employeeId = session('employee_id');

            // Prepare the request data
            $data = [
                'employee_id' => $employeeId,
                'org_id' => $org_id,
                'company_name' => $request->company_name,
                'designation' => $request->designation,
                'joining_date' => $request->joining_date,
                'leaving_date' => $request->leaving_date,
                'reason' => $request->reason,
            ];

            // Send the PUT request to the API to update work history
            $response = Http::withToken($token)->put("{$apiBaseUrl}/work_history/update/{$id}", $data);

            if ($response->successful()) {
                return redirect()->back()->with('success', 'Experience updated successfully');
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update Work History'
                ], $response->status());
            }
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Something went wrong: ' . $e->getMessage()], 500);
        }
    }
}
