<?php

namespace App\Http\Controllers\SuperAdmin;

use CURLFile;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;
use App\DataTables\SuperAdmin\OrganizationDataTable;

class OrganizationController extends Controller
{
    private function fetch_users(): array
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->get("{$apiBaseUrl}/users?search=admin_normal");

        if ($response->successful()) {
            $users = $response->json()['users'] ?? [];
            // Extract only the necessary columns
            $users = array_map(function ($users) {
                return [
                    'id' => $users['id'],
                    'name' => $users['name'],
                ];
            }, $users);

            return $users;
        }

        return [];
    }
    public function organization_list(OrganizationDataTable $dataTable)
    {
        $users = $this->fetch_users();
        return $dataTable->render('super_admin.organization.organization', compact('users'));
    }
    public function add_organization_form(Request $request)
    {
        try {
            // Validate the incoming request
            $validatedData = $request->validate([
                'name' => 'required|string|max:255',
                'permissions' => 'required|array|min:1', // Ensure permissions is an array
                'user_id' => 'required|integer', // Ensure user_id exists in users table
                'image' => 'nullable|file|mimes:jpeg,png,jpg,pdf,doc,docx|max:2048', // Image is optional
            ], [
                'permissions.required' => 'Please select at least one permission',
                'user_id.required' => 'Please select a user',
                'image.mimes' => 'The image must be a file of type: jpeg, png, jpg, pdf, doc, docx.',
                'image.max' => 'The image may not be greater than 2MB.',
            ]);
    
            // Retrieve the bearer token from the session
            $token = session('bearer_token');
    
            // Prepare the request data
            $requestBody = [
                'name' => $validatedData['name'],
                'permissions[]' => $validatedData['permissions'], // Permissions will be an array
                'user_id' => $validatedData['user_id'],
                'status' => 'active',
            ];
    
      
            $ch = curl_init("https://microsols.com.au/devioo/public/api/organizations/store");
    
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $token,
                'Content-Type: multipart/form-data',
            ]);
    
       
            $postFields = $requestBody;
    
            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $postFields['image'] = new CURLFile(
                    $image->getRealPath(),
                    $image->getMimeType(),
                    $image->getClientOriginalName()
                );
            }
    
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $postFields);
    
       
            $response = curl_exec($ch);
            $responseCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
    
 
            $responseArray = json_decode($response, true);
            
        
            // dd($responseArray);
    
            if ($responseCode === 201) {
                return response()->json(['success' => true, 'message' => 'Organization added successfully']);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => $responseArray['message'] ?? 'Failed to add Organization.'
                ], $responseCode);
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Handle validation errors
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }
    
    

    public function update_organization_form(Request $request, $id)
    {
        try {
            // Validate the incoming request
            $validatedData = $request->validate([
                'name' => 'required|string|max:255',
                'permissions' => 'required|array|min:1',
                'image' => 'nullable|file|mimes:jpeg,png,jpg,pdf,doc,docx|max:2048',
                'status' => 'required|string',
            ], [
                'name.required' => 'The name field is required',
                'permissions.required' => 'Please select at least one permission',
                'image.mimes' => 'The image must be a file of type: jpeg, png, jpg.',
                'image.max' => 'The image may not be greater than 2MB.',
            ]);
    
            $token = session('bearer_token');
    
            // Prepare the request data
            $requestBody = [
                'name' => $validatedData['name'],
                'permissions[]' => $validatedData['permissions'],
                'status' => $validatedData['status'],
            ];
    
            // Handle the image file
            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $requestBody['image'] = new \CURLFile(
                    $image->getRealPath(),
                    $image->getMimeType(),
                    $image->getClientOriginalName()
                );
            }
    
            // Initialize the cURL session
            $ch = curl_init("https://microsols.com.au/devioo/public/api/organizations/update/{$id}");
    
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $token,
                'Content-Type: multipart/form-data',
            ]);
    
            // Use POST to send data
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $requestBody);
    
            // Execute the cURL request
            $response = curl_exec($ch);
            $responseCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            
            if (curl_errno($ch)) {
                throw new \Exception(curl_error($ch));
            }
            
            curl_close($ch);
    
            $responseArray = json_decode($response, true);
            // dd($responseArray);
            if ($responseCode === 200) {
                return response()->json(['success' => true, 'message' => 'Organization updated successfully']);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => $responseArray['message'] ?? 'Failed to update Organization.'
                ], $responseCode);
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }
    public function update_organization_status_form(Request $request, $id)
    {
        try {


            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $response = Http::withToken($token)->post("{$apiBaseUrl}/organizations/update/{$id}", [
                'status' => $request->status,
            ]);

            if ($response->successful()) {
                return response()->json(['success' => true, 'message' => 'Organization status updated  successfully']);
            } else {
                $error = $response->json();
                return response()->json([
                    'success' => false,
                    'message' => $error['message'] ?? 'Failed to update the organization.'
                ], $response->status());
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage()
            ], 500);
        }
    }

    public function delete_organization($id)
    {
        $token = session('bearer_token');
        $apiBaseUrl = env('API_BASE_URL');
        $response = Http::withToken($token)->delete("{$apiBaseUrl}/organizations/{$id}");

        if ($response->successful()) {
            return response()->json(['success' => true, 'message' => 'Organization deleted successfully']);
        } else {
            // Extract error message from the API response, defaulting to a generic message
            $error = $response->json();
            $errorMessage = $error['message'] ?? 'Unable to delete Organization';

            return response()->json([
                'success' => false,
                'message' => $errorMessage
            ], $response->status());
        }
    }


    public function get_organization($id)
    {
        try {
            $token = session('bearer_token');
            $apiBaseUrl = env('API_BASE_URL');
            $response = Http::withToken($token)->get("{$apiBaseUrl}/organizations/get/{$id}");



            if ($response->successful()) {
                // Return the department data from the API response
                return response()->json($response->json()['organization']);
            } else {
                return response()->json(['status' => 404, 'message' => 'Organization not found.'], 404);
            }
        } catch (\Exception $e) {
            // Dump the exception message and die
            dd('An error occurred: ' . $e->getMessage());
            return response()->json(['message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
}
